// +build integration

package test

import (
	"bytes"
	"encoding/json"
	"fmt"
	"log"
	"os"
	"os/exec"
	"strings"
	"testing"
	"time"

	"code.justin.tv/systems/sandstorm/manager"
	"code.justin.tv/systems/sandstorm/testutil"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/aws/credentials/stscreds"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/sts"

	. "github.com/smartystreets/goconvey/convey"
)

type TestConfig struct {
	KeyID     string `json:"keyID"`
	TableName string `json:"tableName"`
	Region    string `json:"region"`
	RoleARN   string `json:"roleArn"`
	TopicARN  string `json:"topicArn"`
}

var config *TestConfig

var classRestricted = fmt.Sprintf("%d", manager.ClassRestricted)
var classCustomer = fmt.Sprintf("%d", manager.ClassCustomer)

const (
	commandName     = "sandstorm-admin"
	optionClass     = "--class"
	optionDNB       = "--doNotBroadcast"
	optionPlaintext = "--plaintext"
	arnRole         = "arn:aws:iam::734326455073:role/sandstorm-apiserver-testing"
)

func createTestManager() *manager.Manager {
	config, err := testutil.LoadTestConfigFromFile("../../../test.hcl")
	So(err, ShouldBeNil)
	So(config, ShouldNotBeNil)
	awsConfig := &aws.Config{
		Region: aws.String(config.Sandstorm.Region),
	}
	stsclient := sts.New(session.New(awsConfig))
	arp := &stscreds.AssumeRoleProvider{
		Duration:     900 * time.Second,
		ExpiryWindow: 10 * time.Second,
		RoleARN:      config.Sandstorm.RoleArn,
		Client:       stsclient,
	}
	awsConfig.WithCredentials(credentials.NewCredentials(arp))
	return manager.New(manager.Config{
		AWSConfig: awsConfig,
		TableName: config.Sandstorm.TableName,
		KeyID:     config.Sandstorm.KeyID,
	})
}

func readConfig() *TestConfig {
	if config != nil {
		return config
	}
	conf, err := os.Open("config.json")
	if err != nil {
		log.Fatal(err.Error())
	}
	decoder := json.NewDecoder(conf)
	config := &TestConfig{}
	err = decoder.Decode(&config)
	if err != nil {
		log.Fatal(err)
	}
	return config
}

func runCliCommand(args []string) (stdout *bytes.Buffer, stderr *bytes.Buffer, err error) {

	_, err = exec.LookPath(commandName)
	if err != nil {
		log.Fatal(fmt.Printf("Cannot find %s \n", commandName))
	}

	cmd := exec.Command(commandName, args...)
	cmd.Env = append(os.Environ(), fmt.Sprintf("AWS_ROLE_ARN=%s", arnRole))
	stdout = bytes.NewBuffer(nil)
	stderr = bytes.NewBuffer(nil)

	cmd.Stdout = stdout
	cmd.Stderr = stderr

	err = cmd.Run()
	return
}

func getSecretValue(secretName string) (stdout string, stderr string, err error) {

	subCommandGetSecret := "get-secret"
	testConfig := readConfig()
	getCommand := []string{subCommandGetSecret, testConfig.TableName, secretName}
	stdoutBuffer, stderrBuffer, err := runCliCommand(getCommand)
	stdout = string(stdoutBuffer.Bytes())
	stderr = string(stderrBuffer.Bytes())
	return
}

func getSecretClass(secretName string) (manager.SecretClass, error) {
	secret, err := getSecretEncrypted(secretName)
	if err != nil {
		return 0, err
	}
	// Extract secret class from output.
	return secret.Class, nil
}

func getSecretDoNotBroadcastFlag(secretName string) (bool, error) {
	secret, err := getSecretEncrypted(secretName)
	if err != nil {
		return false, err
	}
	// Extract secret DNB flag from output.
	return secret.DoNotBroadcast, nil
}

func getSecretEncrypted(secretName string) (*manager.Secret, error) {
	subCommandGetSecret := "get-secret-encrypted"
	testConfig := readConfig()
	getCommand := []string{subCommandGetSecret, testConfig.TableName, secretName}
	contents, _, err := runCliCommand(getCommand)
	if err != nil {
		return nil, err
	}

	secret := new(manager.Secret)
	err = json.Unmarshal(contents.Bytes(), &secret)
	if err != nil {
		fmt.Println(err.Error())
		return nil, err
	}

	return secret, nil
}

func deleteSecret(secretName string) {
	subCommand := "delete-secret"
	testConfig := readConfig()
	getCommand := []string{subCommand, testConfig.TableName, secretName}
	_, _, err := runCliCommand(getCommand)
	if err != nil {
		log.Println(err.Error())
	}

}

func getRandomSecretName() string {
	return testutil.GetRandomSecretNameWithPrefix("admin-test")
}

func getCreateSecretBaseArgs(secretName string, secretValue string) []string {
	testConfig := readConfig()
	return []string{"create-secret", testConfig.TableName, testConfig.KeyID, secretName, secretValue}
}

func getUpdateSecretBaseArgs(secretName string) []string {
	testConfig := readConfig()
	return []string{"update-secret", testConfig.TableName, testConfig.KeyID, secretName}
}

func TestSandstormAdminCreateSecret(t *testing.T) {
	testConfig := readConfig()

	Convey("Test create a secret using sandstorm-admin", t, func() {
		secretName := getRandomSecretName()
		secretValue := "someSecret"
		createSubcommand := getCreateSecretBaseArgs(secretName, secretValue)
		Convey("Config should have been read correctly", func() {
			So(testConfig, ShouldNotBeNil)
		})

		Convey("should pass when class or doNotBroadcast is not specified.", func() {
			out, _, err := runCliCommand(createSubcommand)
			So(err, ShouldBeNil)
			So(out, ShouldNotBeNil)
			storedSecret, _, err := getSecretValue(secretName)
			So(err, ShouldBeNil)
			So(storedSecret, ShouldEqual, secretValue)
		})

		Convey("should pass when class was provided.", func() {
			args := append(createSubcommand, []string{optionClass, classRestricted}...)
			out, _, err := runCliCommand(args)
			So(err, ShouldBeNil)
			So(out, ShouldNotBeNil)

			secretClass, err := getSecretClass(secretName)
			So(err, ShouldBeNil)
			So(secretClass, ShouldEqual, manager.ClassRestricted)
		})

		Convey("should pass when doNotBroadast flag was provided.", func() {

			args := append(createSubcommand, optionDNB)
			out, _, err := runCliCommand(args)
			So(err, ShouldBeNil)
			So(out, ShouldNotBeNil)

			dnb, err := getSecretDoNotBroadcastFlag(secretName)
			So(err, ShouldBeNil)
			So(dnb, ShouldBeTrue)
		})

		Convey("should pass when class and doNotBroadast flag was provided.", func() {
			args := append(createSubcommand, []string{optionClass, classRestricted, optionDNB}...)
			out, _, err := runCliCommand(args)
			So(err, ShouldBeNil)
			So(out, ShouldNotBeNil)

			dnb, err := getSecretDoNotBroadcastFlag(secretName)
			So(err, ShouldBeNil)
			So(dnb, ShouldBeTrue)

			secretClass, err := getSecretClass(secretName)
			So(err, ShouldBeNil)
			So(secretClass, ShouldEqual, manager.ClassRestricted)
		})

		Reset(func() {
			deleteSecret(secretName)
		})
	})
}

func TestSandstormAdminUpdateSecret(t *testing.T) {
	testConfig := readConfig()
	secretName := getRandomSecretName()
	secretValue := "someSecret"
	createSubcommand := getCreateSecretBaseArgs(secretName, secretValue)

	Convey("Test Update of an existing secret using sandstorm-admin", t, func() {
		out, _, err := runCliCommand(createSubcommand)

		So(err, ShouldBeNil)
		So(out, ShouldNotBeNil)

		subCommandUpdateSecret := "update-secret"
		updateSubcommand := []string{subCommandUpdateSecret, testConfig.TableName, testConfig.KeyID, secretName}

		Convey("should pass when class is specified.", func() {
			args := append(updateSubcommand, []string{optionClass, classCustomer}...)
			out, _, err := runCliCommand(args)
			So(err, ShouldBeNil)
			So(out, ShouldNotBeNil)
			secretClass, err := getSecretClass(secretName)
			So(err, ShouldBeNil)
			So(secretClass, ShouldEqual, manager.ClassCustomer)
		})

		Convey("should pass when doNotBroadcast flag is specified.", func() {
			args := append(updateSubcommand, optionDNB)
			out, _, err := runCliCommand(args)
			So(err, ShouldBeNil)
			So(out, ShouldNotBeNil)
			secretClass, err := getSecretDoNotBroadcastFlag(secretName)
			So(err, ShouldBeNil)
			So(secretClass, ShouldBeTrue)
		})

		Convey("should pass when class and doNotBroadcast is specified.", func() {
			args := append(updateSubcommand, []string{optionClass, classCustomer, optionDNB}...)
			out, _, err := runCliCommand(args)
			So(err, ShouldBeNil)
			So(out, ShouldNotBeNil)

			secretClass, err := getSecretClass(secretName)
			So(err, ShouldBeNil)
			So(secretClass, ShouldEqual, manager.ClassCustomer)
			dnbFlag, err := getSecretDoNotBroadcastFlag(secretName)
			So(err, ShouldBeNil)
			So(dnbFlag, ShouldBeTrue)

		})

		Convey("should pass when plaintext is specified.", func() {
			newSecret := "AnotherSecret"
			args := append(updateSubcommand, []string{optionPlaintext, newSecret}...)
			out, _, err := runCliCommand(args)
			So(err, ShouldBeNil)
			So(out, ShouldNotBeNil)
			updatedSecret, _, err := getSecretValue(secretName)
			So(err, ShouldBeNil)
			So(updatedSecret, ShouldEqual, newSecret)
		})

		Convey("Should pass when plaintext and class is specified.", func() {
			newSecret := "AnotherSecret"
			args := append(updateSubcommand, []string{optionPlaintext, newSecret, optionClass, classCustomer}...)
			out, _, err := runCliCommand(args)
			So(err, ShouldBeNil)
			So(out, ShouldNotBeNil)

			updatedSecret, _, err := getSecretValue(secretName)
			So(err, ShouldBeNil)
			So(updatedSecret, ShouldEqual, newSecret)

			updatedClass, err := getSecretClass(secretName)
			So(err, ShouldBeNil)
			So(updatedClass, ShouldEqual, manager.ClassCustomer)
		})

		Convey("should pass when plaintext and dnb flag is specified.", func() {
			newSecret := "AnotherSecret"
			args := append(updateSubcommand, []string{optionPlaintext, newSecret, optionDNB}...)
			out, _, err := runCliCommand(args)
			So(err, ShouldBeNil)
			So(out, ShouldNotBeNil)
			updatedSecret, _, err := getSecretValue(secretName)
			So(err, ShouldBeNil)
			So(updatedSecret, ShouldEqual, newSecret)

			dnb, err := getSecretDoNotBroadcastFlag(secretName)
			So(err, ShouldBeNil)
			So(dnb, ShouldBeTrue)
		})

		Reset(func() {
			deleteSecret(secretName)
		})
	})

}

func TestSandstormAdminGetVersions(t *testing.T) {
	testConfig := readConfig()

	Convey("get-secret-versions", t, func() {
		m := createTestManager()

		Convey("key exists with versions", func() {
			secretName := getRandomSecretName()
			So(m.Post(&manager.Secret{
				Name:      secretName,
				Plaintext: []byte("someSecret"),
			}), ShouldBeNil)
			versions, err := m.GetVersionsEncrypted(secretName, 10, 0)
			So(err, ShouldBeNil)

			out, _, err := runCliCommand([]string{"get-secret-versions", testConfig.TableName, secretName})
			responseStr := string(out.Bytes())
			So(err, ShouldBeNil)
			So(responseStr, ShouldEqual, fmt.Sprintf("%d\n", versions.Secrets[0].UpdatedAt))

			Reset(func() {
				So(m.Delete(secretName), ShouldBeNil)
			})
		})

		Convey("key does not exist", func() {
			secretName := getRandomSecretName()
			out, _, err := runCliCommand([]string{"get-secret-versions", testConfig.TableName, secretName})
			responseStr := string(out.Bytes())
			t.Logf(responseStr)
			So(err, ShouldBeNil)
			So(responseStr, ShouldEqual, "")
		})
	})
}

func TestSandstormAdminRevert(t *testing.T) {
	testConfig := readConfig()
	secretValues := [][]byte{
		[]byte{0x50},
		[]byte{0x51},
	}

	Convey("revert-secret", t, func() {
		m := createTestManager()
		secretName := getRandomSecretName()
		So(m.Post(&manager.Secret{
			Name:      secretName,
			Plaintext: secretValues[0],
		}), ShouldBeNil)
		// Versions have 1s accuracy
		time.Sleep(1 * time.Second)
		So(m.Patch(&manager.PatchInput{
			Name:      secretName,
			Plaintext: secretValues[1],
		}), ShouldBeNil)

		versions, err := m.GetVersionsEncrypted(secretName, 10, 0)
		So(err, ShouldBeNil)
		So(len(versions.Secrets), ShouldEqual, 2)

		Convey("version exists", func() {
			secret := versions.Secrets[1]
			out, _, err := runCliCommand([]string{
				"revert-secret",
				testConfig.TableName,
				secretName,
				fmt.Sprintf("%d", secret.UpdatedAt),
			})
			outStr := string(out.Bytes())
			t.Logf(outStr)
			So(err, ShouldBeNil)
		})

		Convey("version does not exist", func() {
			_, stderr, err := runCliCommand([]string{
				"revert-secret",
				testConfig.TableName,
				secretName,
				fmt.Sprintf("%d", versions.Secrets[0].UpdatedAt+1),
			})

			errStr := string(stderr.Bytes())
			So(err, ShouldNotBeNil)
			So(strings.HasSuffix(errStr, "secret version does not exist\n"), ShouldBeTrue)
		})

		Reset(func() {
			So(m.Delete(secretName), ShouldBeNil)
		})
	})
}

func TestSandstormAdminForInvalidCommands(t *testing.T) {
	Convey("Create secret failes for an existing secret when setting --class to ", t, func() {
		secretName := getRandomSecretName()
		secretValue := "someSecret"
		createSubcommand := getCreateSecretBaseArgs(secretName, secretValue)
		Convey("invalid option: 6", func() {
			args := append(createSubcommand, []string{optionClass, "6"}...)
			_, stderr, err := runCliCommand(args)
			So(err, ShouldNotBeNil)
			So(stderr.String(), ShouldContainSubstring, "--class can only take values 1 to 4")
		})

		Convey("invalid option: 0", func() {
			args := append(createSubcommand, []string{optionClass, "0"}...)
			_, stderr, err := runCliCommand(args)
			So(err, ShouldNotBeNil)
			So(stderr.String(), ShouldContainSubstring, "--class can only take values 1 to 4")
		})
	})

	Convey("Update secret failes to update a non existing secret", t, func() {
		secretName := getRandomSecretName()
		updateSubcommand := getUpdateSecretBaseArgs(secretName)
		args := append(updateSubcommand, []string{optionClass, classRestricted}...)
		_, stderr, err := runCliCommand(args)
		So(err, ShouldNotBeNil)
		expectedError := fmt.Sprintf("Secret %s not found. Please use 'POST' to create a secret", secretName)
		So(stderr.String(), ShouldContainSubstring, expectedError)
	})

	Convey("Create secret failes to create an existing secret", t, func() {
		secretName := getRandomSecretName()
		secretValue := "SomeSecret"
		createSubcommand := getCreateSecretBaseArgs(secretName, secretValue)

		Convey("Create a new secret", func() {
			_, _, err := runCliCommand(createSubcommand)
			So(err, ShouldBeNil)

			actualSecret, _, err := getSecretValue(secretName)
			So(err, ShouldBeNil)
			So(actualSecret, ShouldEqual, secretValue)

			_, stderr, err := runCliCommand(createSubcommand)
			So(err, ShouldNotBeNil)
			expectedError := fmt.Sprintf("Secret %s already found. Please use 'PATCH' to update a secret", secretName)
			So(stderr.String(), ShouldContainSubstring, expectedError)
		})

		Reset(func() {
			deleteSecret(secretName)
		})
	})
}
