// +build integration

package heartbeat

import (
	"context"
	"errors"
	"testing"
	"time"

	"code.justin.tv/systems/sandstorm/testutil"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/aws/credentials/stscreds"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbattribute"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbiface"
	"github.com/aws/aws-sdk-go/service/iam"
	"github.com/aws/aws-sdk-go/service/iam/iamiface"
	"github.com/aws/aws-sdk-go/service/sts"
	uuid "github.com/satori/go.uuid"
	"github.com/stretchr/testify/assert"
)

const integrationTestServiceName = "myService"
const integrationTestTableName = "heartbeats-testing"
const integrationTestHostName = "myHostName"
const integrationTestFQDN = "myFQDN"
const integrationTestTTL = 259200

type clientIntegrationTest struct {
	client     *Client
	db         dynamodbiface.DynamoDBAPI
	iam        iamiface.IAMAPI
	secretName string
	callerArn  string
}

func (cit *clientIntegrationTest) getUserArn(t *testing.T) (userArn string) {
	return cit.callerArn
}

func (cit *clientIntegrationTest) getCompositeKey(t *testing.T) (key string) {
	key = cit.getUserArn(t)
	key += ":" + integrationTestHostName
	key += ":" + integrationTestServiceName
	key += ":" + cit.secretName
	return
}

func (cit *clientIntegrationTest) deleteHeartbeat(t *testing.T) {
	_, err := cit.db.DeleteItem(&dynamodb.DeleteItemInput{
		TableName: aws.String(integrationTestTableName),
		Key: map[string]*dynamodb.AttributeValue{
			"composite_key": {S: aws.String(cit.getCompositeKey(t))},
		},
	})

	if err != nil {
		t.Fatal(err)
	}
}

func (cit *clientIntegrationTest) assertHeartbeatEmpty(t *testing.T) {
	assert.Nil(t, cit.getHeartbeat(t))
}

func (cit *clientIntegrationTest) getHeartbeat(t *testing.T) (hb *DynamoHeartbeat) {
	output, err := cit.db.GetItem(&dynamodb.GetItemInput{
		TableName: aws.String(integrationTestTableName),
		Key: map[string]*dynamodb.AttributeValue{
			"composite_key": {S: aws.String(cit.getCompositeKey(t))},
		},
	})

	if err != nil {
		t.Fatal(err)
	}

	if output.Item == nil {
		return
	}

	hb = new(DynamoHeartbeat)
	err = dynamodbattribute.UnmarshalMap(output.Item, hb)
	if err != nil {
		t.Fatal(err)
	}

	return
}

func newClientIntegrationTest(t *testing.T) (cit *clientIntegrationTest) {
	testConfig, err := testutil.LoadTestConfigFromFile("../../test.hcl")
	if err != nil {
		t.Fatal(err)
	}

	heartbeatConfig := &Config{
		URL:     testConfig.Sandstorm.InventoryStatusURL,
		Service: integrationTestServiceName,
		Host:    integrationTestHostName,
		FQDN:    integrationTestFQDN,
		Region:  testConfig.Sandstorm.Region,
	}

	awsConfig := &aws.Config{
		Region: aws.String(testConfig.Sandstorm.Region),
	}

	stsclient := sts.New(session.New(awsConfig))
	arp := &stscreds.AssumeRoleProvider{
		Duration:     900 * time.Second,
		ExpiryWindow: 10 * time.Second,
		RoleARN:      testConfig.Sandstorm.InventoryAdminRoleARN,
		Client:       stsclient,
	}
	creds := credentials.NewCredentials(arp)
	awsConfig.WithCredentials(creds)

	sess := session.New(awsConfig)

	stsclient = sts.New(sess)
	identity, err := stsclient.GetCallerIdentity(&sts.GetCallerIdentityInput{})
	if err != nil {
		t.Fatal(err)
	}

	return &clientIntegrationTest{
		client:     New(creds, heartbeatConfig, nil),
		db:         dynamodb.New(sess),
		iam:        iam.New(sess),
		secretName: uuid.NewV4().String(),
		callerArn:  aws.StringValue(identity.Arn),
	}
}

func (cit *clientIntegrationTest) teardown(t *testing.T) {
	cit.deleteHeartbeat(t)
}

func TestClientIntegration(t *testing.T) {

	t.Run("putHeartbeat", func(t *testing.T) {
		// this will be set in intitialize secret heartbeat
		var firstRetrieval int64

		cit := newClientIntegrationTest(t)
		defer cit.teardown(t)

		t.Run("initial state", func(t *testing.T) {
			cit.assertHeartbeatEmpty(t)
		})

		t.Run("initialize secret heartbeat", func(t *testing.T) {
			const secretUpdatedAt = 101

			assert := assert.New(t)

			cit.client.UpdateHeartbeat(&Secret{
				Name:      cit.secretName,
				UpdatedAt: secretUpdatedAt,
			})

			firstRetrieval = cit.client.heartbeatState.secretsToReport[cit.secretName].FetchedAt

			err := cit.client.putHeartbeat()
			assert.NoError(err)

			hb := cit.getHeartbeat(t)
			if hb == nil {
				t.Fatal(errors.New("unexpected empty response"))
			}

			receivedAt := hb.HeartbeatReceived
			assert.Equal(DynamoHeartbeat{
				CompositeKey:      cit.getCompositeKey(t),
				Service:           integrationTestServiceName,
				Host:              integrationTestHostName,
				FQDN:              integrationTestFQDN,
				Secret:            cit.secretName,
				UpdatedAt:         secretUpdatedAt,
				FetchedAt:         firstRetrieval,
				FirstRetrievedAt:  firstRetrieval,
				ExpiresAt:         receivedAt + integrationTestTTL,
				HeartbeatReceived: receivedAt,
			}, *hb)
		})

		t.Run("updated heartbeat", func(t *testing.T) {
			const secretUpdatedAt = 105

			assert := assert.New(t)

			cit.client.UpdateHeartbeat(&Secret{
				Name:      cit.secretName,
				UpdatedAt: secretUpdatedAt,
			})

			fetchedAt := cit.client.heartbeatState.secretsToReport[cit.secretName].FetchedAt

			err := cit.client.putHeartbeat()
			assert.NoError(err)

			hb := cit.getHeartbeat(t)
			if hb == nil {
				t.Fatal(errors.New("unexpected empty response"))
			}

			heartbeatReceived := hb.HeartbeatReceived

			assert.Equal(DynamoHeartbeat{
				CompositeKey:      cit.getCompositeKey(t),
				Service:           integrationTestServiceName,
				Host:              integrationTestHostName,
				FQDN:              integrationTestFQDN,
				Secret:            cit.secretName,
				UpdatedAt:         secretUpdatedAt,
				FetchedAt:         fetchedAt,
				FirstRetrievedAt:  firstRetrieval,
				HeartbeatReceived: heartbeatReceived,
				ExpiresAt:         heartbeatReceived + integrationTestTTL,
			}, *hb)
		})
	})

	t.Run("start", func(t *testing.T) {
		// this will be set in first fetch
		var firstRetrieval int64

		cit := newClientIntegrationTest(t)
		defer cit.teardown(t)

		cit.client.Config.Interval = 100 * time.Millisecond

		defer func() {
			assert.NoError(t, cit.client.Stop())
		}()
		go cit.client.Start()

		t.Run("heartbeat should not be pushed if empty", func(t *testing.T) {
			cit.assertHeartbeatEmpty(t)
		})

		t.Run("heartbeat should be initialized on UpdateHeartbeat", func(t *testing.T) {
			assert := assert.New(t)
			const secretUpdatedAt = 177
			cit.client.UpdateHeartbeat(&Secret{
				Name:      cit.secretName,
				UpdatedAt: secretUpdatedAt,
			})

			firstRetrieval = cit.client.heartbeatState.secretsToReport[cit.secretName].FetchedAt

			var checked = false
			for nTry := 0; nTry < 50; nTry++ {
				time.Sleep(100 * time.Millisecond)

				hb := cit.getHeartbeat(t)
				if hb == nil {
					continue
				}
				heartbeatReceived := hb.HeartbeatReceived

				checked = true
				assert.Equal(DynamoHeartbeat{
					CompositeKey:      cit.getCompositeKey(t),
					Service:           integrationTestServiceName,
					Host:              integrationTestHostName,
					FQDN:              integrationTestFQDN,
					Secret:            cit.secretName,
					UpdatedAt:         secretUpdatedAt,
					HeartbeatReceived: heartbeatReceived,
					FetchedAt:         firstRetrieval,
					FirstRetrievedAt:  firstRetrieval,
					ExpiresAt:         heartbeatReceived + integrationTestTTL,
				}, *hb)
				break
			}

			assert.True(checked)
		})

		var updateHeartbeatTime int64

		t.Run("heartbeat should be updated on UpdateHeartbeat", func(t *testing.T) {
			assert := assert.New(t)
			const secretUpdatedAt = 197
			cit.client.UpdateHeartbeat(&Secret{
				Name:      cit.secretName,
				UpdatedAt: secretUpdatedAt,
			})

			currentFetchedAt := cit.client.heartbeatState.secretsToReport[cit.secretName].FetchedAt

			var checked = false
			for nTry := 0; nTry < 50; nTry++ {
				time.Sleep(100 * time.Millisecond)

				hb := cit.getHeartbeat(t)
				if hb == nil || secretUpdatedAt != hb.UpdatedAt {
					continue
				}

				heartbeatReceived := hb.HeartbeatReceived
				updateHeartbeatTime = heartbeatReceived

				checked = true
				assert.Equal(DynamoHeartbeat{
					CompositeKey:      cit.getCompositeKey(t),
					Service:           integrationTestServiceName,
					Host:              integrationTestHostName,
					FQDN:              integrationTestFQDN,
					Secret:            cit.secretName,
					UpdatedAt:         secretUpdatedAt,
					FetchedAt:         currentFetchedAt,
					FirstRetrievedAt:  firstRetrieval,
					HeartbeatReceived: heartbeatReceived,
					ExpiresAt:         heartbeatReceived + integrationTestTTL,
				}, *hb)
				break
			}

			assert.True(checked)
		})

		t.Run("heartbeat should be flushed on SendHeartbeat", func(t *testing.T) {
			assert := assert.New(t)
			const secretUpdatedAt = 197

			currentFetchedAt := cit.client.heartbeatState.secretsToReport[cit.secretName].FetchedAt

			ctx, cancel := context.WithTimeout(context.Background(), time.Second)
			defer cancel()
			cit.client.FlushHeartbeat(ctx)

			var checked = false
			for nTry := 0; nTry < 50; nTry++ {
				time.Sleep(100 * time.Millisecond)

				hb := cit.getHeartbeat(t)
				if hb == nil || secretUpdatedAt != hb.UpdatedAt {
					continue
				}

				heartbeatReceived := hb.HeartbeatReceived

				if heartbeatReceived == updateHeartbeatTime {
					continue
				}

				checked = true
				assert.True(heartbeatReceived > updateHeartbeatTime, "flushed heartbeat should be after updated heartbeat")
				assert.Equal(DynamoHeartbeat{
					CompositeKey:      cit.getCompositeKey(t),
					Service:           integrationTestServiceName,
					Host:              integrationTestHostName,
					FQDN:              integrationTestFQDN,
					Secret:            cit.secretName,
					UpdatedAt:         secretUpdatedAt,
					FetchedAt:         currentFetchedAt,
					FirstRetrievedAt:  firstRetrieval,
					HeartbeatReceived: heartbeatReceived,
					ExpiresAt:         heartbeatReceived + integrationTestTTL,
				}, *hb)
				break
			}

			assert.True(checked)
		})
	})
}
