""" inventory/scripts/admin.py

This script admininstrates the inventory deployment. It expects the following
paths:

- lambda.zip: the lambda to deploy

"""
from collections import namedtuple
import argparse
import logging
import os

import boto3

LOG = logging.getLogger('admin')

LAMBDA_ARTIFACT = 'lambda.zip'
DEPLOYMENT_ROLE_ARN = 'arn:aws:iam::516651178292:role/inventory-lambda-artifact-deploy'  # NOQA
S3_BUCKET = 'sse-inventory'

environment = namedtuple('environment', (
    'name',
    'account_id',
    'update_lambda_role_arn',
))

ENVIRONMENTS = (
    environment(
        account_id='516651178292',
        name='testing',
        update_lambda_role_arn='arn:aws:iam::516651178292:role/inventory-lambda-deploy-testing', # NOQA
    ),
    environment(
        account_id='516651178292',
        name='staging',
        update_lambda_role_arn='arn:aws:iam::516651178292:role/inventory-lambda-deploy-staging', # NOQA
    ),
    environment(
        account_id='854594403332',
        name='production',
        update_lambda_role_arn='arn:aws:iam::854594403332:role/inventory-lambda-deploy-production', # NOQA
    ),
)


def deploy(env):
    """ Deploy lambdas to an environment.

    :param environment env: Environment to deploy to.
    """
    LOG.info('deploying lambda to environment {}'.format(env))
    session = _session(DEPLOYMENT_ROLE_ARN)
    with open(LAMBDA_ARTIFACT, 'r') as f:
        object_version = _deploy_s3(session, env, f)
        _deploy_lambda_update_fn(session, env, object_version)


def _deploy_s3(session, env, body):
    """ Upload zip to S3 and return the object version.
    """
    LOG.info('uploading to s3')
    info = session.client('s3').put_object(
        Body=body,
        Bucket=S3_BUCKET,
        Key=_s3_key(env),
    )
    return info['VersionId']


def _deploy_lambda_update_fn(session, env, object_version):
    """ Trigger lambda function update from the latest version.
    """
    LOG.info('updating lambda function')
    session = _session(
        'arn:aws:iam::{}:role/inventory-lambda-deploy-{}'.format(
            env.account_id, env.name),
        session)

    session.client('lambda').update_function_code(
        FunctionName=_lambda_fn_name(env),
        Publish=True,
        S3Bucket=S3_BUCKET,
        S3Key=_s3_key(env),
    )


def _session(role_arn, session=None, region='us-west-2'):
    if session is None:
        LOG.info('creating new session')
        session = boto3.session.Session(
            region_name=region,
        )
    LOG.info('assuming role: {}'.format(role_arn))
    sts = session.client('sts')
    creds = sts.assume_role(
        RoleArn=role_arn,
        RoleSessionName='inventory-admin',
    )['Credentials']
    return boto3.session.Session(
        aws_access_key_id=creds['AccessKeyId'],
        aws_secret_access_key=creds['SecretAccessKey'],
        aws_session_token=creds['SessionToken'],
        region_name=region,
    )


def _s3_key(env):
    s3_key = os.path.join('lambdas', env.name + '.zip')
    LOG.info('s3_key: {}'.format(s3_key))
    return s3_key


def _lambda_fn_name(env):
    name = 'put_heartbeat-{}'.format(env.name)
    LOG.info('lambda fn name: {}'.format(name))
    return name


def main():
    ap = argparse.ArgumentParser('admin.py')
    sp = ap.add_subparsers(help='commands', dest='command')

    envs = {x.name: x for x in ENVIRONMENTS}

    p = sp.add_parser('deploy')
    p.add_argument(
        'environment',
        choices=sorted(envs.keys()),
    )

    args = ap.parse_args()

    logging.basicConfig(
        level='INFO',
    )

    if args.command == 'deploy':
        deploy(envs[args.environment])


if __name__ == '__main__':
    main()
