resource "aws_iam_policy" "sandstorm-rw" {
  provider    = "aws.twitch-sandstorm-infra-aws"
  name_prefix = "${var.name}"
  path        = "/terraform/${var.env}/policy/"
  policy      = "${data.aws_iam_policy_document.sandstorm-rw.json}"
}

resource "aws_iam_role" "sandstorm-rw" {
  provider           = "aws.twitch-sandstorm-infra-aws"
  name_prefix        = "${var.name}"
  path               = "/terraform/${var.env}/role/"
  assume_role_policy = "${data.aws_iam_policy_document.sandstorm-rw-assume-role.json}"
}

resource "aws_iam_policy_attachment" "sandstorm-rw" {
  provider   = "aws.twitch-sandstorm-infra-aws"
  name       = "sandstorm-rw-attachment"
  roles      = ["${aws_iam_role.sandstorm-rw.name}"]
  policy_arn = "${aws_iam_policy.sandstorm-rw.arn}"
}

data "aws_iam_policy_document" "sandstorm-rw-assume-role" {
  statement {
    actions = [
      "sts:AssumeRole",
    ]

    principals {
      type        = "AWS"
      identifiers = ["${var.assume_role_identifiers}"]
    }
  }
}

data "aws_iam_policy_document" "sandstorm-rw" {
  statement {
    sid = "kms"

    actions = [
      "kms:Decrypt",
      "kms:GenerateDataKey",
    ]

    resources = [
      "arn:aws:kms:us-west-2:${var.aws_account_id}:key/${var.kms_key_ids[var.env]}",
    ]
  }

  statement {
    sid = "policies"

    actions = [
      "iam:AttachRolePolicy",
      "iam:CreatePolicy",
      "iam:CreatePolicyVersion",
      "iam:CreateRole",
      "iam:DeletePolicy",
      "iam:DeletePolicyVersion",
      "iam:DeleteRole",
      "iam:DetachRolePolicy",
      "iam:GetPolicy",
      "iam:GetPolicyVersion",
      "iam:GetRole",
      "iam:ListAttachedRolePolicies",
      "iam:ListPolicies",
      "iam:ListPolicyVersions",
      "iam:ListRoles",
      "iam:SetDefaultPolicyVersion",
      "iam:UpdateAssumeRolePolicy",
    ]

    resources = [
      "arn:aws:iam::${var.aws_account_id}:policy/sandstorm/*/templated/policy/",
      "arn:aws:iam::${var.aws_account_id}:policy/sandstorm/*/templated/policy/*",
      "arn:aws:iam::${var.aws_account_id}:policy/sandstorm-*",
      "arn:aws:iam::${var.aws_account_id}:policy/Sandstorm-*",
      "arn:aws:iam::${var.aws_account_id}:role/sandstorm/*/templated/role/",
      "arn:aws:iam::${var.aws_account_id}:role/sandstorm/*/templated/role/*",
      "arn:aws:iam::${var.aws_account_id}:role/sandstorm*",
      "arn:aws:iam::${var.aws_account_id}:role/Sandstorm-*",
      "arn:aws:iam::${var.aws_account_id}:policy/sandstorm/*/aux_policy/",
      "arn:aws:iam::${var.aws_account_id}:policy/sandstorm/*/aux_policy/*",
    ]
  }

  statement {
    sid = "UsersAndGroups"

    actions = [
      "iam:AttachRolePolicy",
      "iam:DetachRolePolicy",
      "iam:GetUser",
      "iam:GetGroup",
      "iam:CreateGroup",
      "iam:AttachGroupPolicy",
      "iam:DetachGroupPolicy",
      "iam:DeleteGroup",
      "iam:AddUserToGroup",
      "iam:RemoveUserFromGroup",
    ]

    resources = [
      "arn:aws:iam::${var.aws_account_id}:policy/*",
      "arn:aws:iam::${var.aws_account_id}:role/*",
      "arn:aws:iam::${var.aws_account_id}:user/sandstorm/*/sandstorm-*",
      "arn:aws:iam::${var.aws_account_id}:user/sandstorm-*",
      "arn:aws:iam::${var.aws_account_id}:user/Sandstorm-*",
      "arn:aws:iam::${var.aws_account_id}:group/sandstorm/*",
      "arn:aws:iam::${var.aws_account_id}:group/Sandstorm-*",
      "arn:aws:iam::${var.aws_account_id}:group/sandstorm-*",
      "arn:aws:iam::${var.aws_account_id}:group/sandstorm*",
    ]
  }

  statement {
    sid = "listIAM"

    actions = [
      "iam:ListUsers",
      "iam:ListGroupsForUser",
      "iam:ListGroups",
      "iam:ListAttachedGroupPolicies",
      "iam:ListAttachedRolePolicies",
    ]

    resources = [
      "arn:aws:iam::${var.aws_account_id}:policy/*",
      "arn:aws:iam::${var.aws_account_id}:user/*",
      "arn:aws:iam::${var.aws_account_id}:group/sandstorm/*",
      "arn:aws:iam::${var.aws_account_id}:role/*",
    ]
  }

  statement {
    sid = "dynamodb"

    actions = [
      "dynamodb:DeleteItem",
      "dynamodb:PutItem",
      "dynamodb:Query",
      "dynamodb:Scan",
      "dynamodb:DescribeTable",
      "dynamodb:UpdateItem",
      "dynamodb:GetItem",
    ]

    resources = [
      "arn:aws:dynamodb:us-west-2:${var.aws_account_id}:table/sandstorm-${var.env}",
      "arn:aws:dynamodb:us-west-2:${var.aws_account_id}:table/sandstorm-${var.env}_audit",
      "arn:aws:dynamodb:us-west-2:${var.aws_account_id}:table/sandstorm-${var.env}_namespaces",
      "arn:aws:dynamodb:us-west-2:${var.aws_account_id}:table/sandstorm-${var.env}_role_owners",
      "arn:aws:dynamodb:us-west-2:${var.aws_account_id}:table/sandstorm-${var.env}/index/namespace_name",
      "arn:aws:dynamodb:us-west-2:${var.aws_account_id}:table/sandstorm-${var.env}_audit/index/namespace_name",
    ]
  }
}
