package main

import (
	"context"
	"flag"
	"fmt"
	"os"
	"reflect"
	"strings"
	"text/template"

	"code.justin.tv/tshadwell/nice/naiive"
	"github.com/google/subcommands"
)

type ListCommand struct {
	Template string
}

func init() { subcommands.Register(&ListCommand{}, "") }

func (ListCommand) Name() string          { return "list" }
func (ListCommand) Synopsis() string      { return "display information about builtin tests" }
func (l *ListCommand) Invocation() string { return invocationReference(l, "", "") }

func (l *ListCommand) Usage() string {
	text := l.Invocation() + `
	Displays information on the builtin tests. Specified information is printed from each test using the go templating format.
Fields:
`

	var fields []string
	t := reflect.TypeOf(naiive.Test{})
	for i, ed := 0, t.NumField(); i < ed; i++ {
		f := t.Field(i)
		fields = append(fields, fmt.Sprintf("%s (%s)", f.Name, f.Type))
	}

	text = text + strings.Join(fields, "\n")

	return text
}

const defaultListTemplate = `{{.Name}} ({{.ShortName}})
Confidence: {{.Confidence}}
Severity: {{.Severity}}

{{.Desc}}

Examples:
{{range .Examples}}
	{{ .}}
{{ end}}


`

func (l *ListCommand) SetFlags(fs *flag.FlagSet) {
	fs.StringVar(&l.Template, "f", defaultListTemplate, "go `template` to specify how information on the tests is formatted and displayed")
}

func (l *ListCommand) Execute(ctx context.Context, fs *flag.FlagSet, args ...interface{}) subcommands.ExitStatus {
	if err := l.execute(ctx, fs, args...); err != nil {
		fmt.Printf("execute error %s\n", err)
		return 1
	}

	return 0
}

func (l *ListCommand) execute(ctx context.Context, f *flag.FlagSet, args ...interface{}) (err error) {
	t, err := template.New("list-format").Parse(l.Template)
	if err != nil {
		return
	}

	for _, test := range naiive.DefaultTests {
		if err = t.Execute(os.Stdout, test); err != nil {
			return
		}
	}

	return
}
