package main

import (
	"context"
	"flag"
	"fmt"
	"regexp"
	"strings"

	"github.com/google/subcommands"

	"code.justin.tv/tshadwell/nice/naiive"
)

type RegexpFlag regexp.Regexp

func (rl *RegexpFlag) String() string {
	return (*regexp.Regexp)(rl).String()
}

func (rl *RegexpFlag) Set(value string) (err error) {
	var re *regexp.Regexp = (*regexp.Regexp)(rl)
	var rep *regexp.Regexp
	if rep, err = regexp.Compile(value); err != nil {
		return
	}

	*re = *rep

	return
}

// this type no longer parses space-separated regexes becuase it was annoying; might alter/replace in future

type RegexpList []*regexp.Regexp

func (rl RegexpList) String() string {
	var out []string = make([]string, 0, len(rl))
	for _, r := range rl {
		if r == nil {
			continue
		}
		out = append(out, r.String())
	}

	return strings.Join(out, " ")
}

func (rl *RegexpList) Set(value string) (err error) {
	//fs := strings.Fields(value)
	fs := []string{value}
	*rl = make([]*regexp.Regexp, len(fs))

	for i, f := range fs {
		if (*rl)[i], err = regexp.Compile(f); err != nil {
			return
		}
	}

	return
}

type SearchCommand struct {
	ScanCommand

	Expressions RegexpList
}

func init() {
	subcommands.Register(&SearchCommand{}, "")
}

func (SearchCommand) Name() string { return "search" }
func (s *SearchCommand) Synopsis() string {
	return `search packages and their dependencies via regular expressions`
}

func (s *SearchCommand) Invocation() string {
	return invocationReference(s, "", "[packages]")
}
func (s *SearchCommand) Usage() string {
	return s.Invocation() + `
   Recursively searches packages and their dependencies for regular expressions.

   Targets use standard package argument syntax; they can be import paths or filesystem paths.
`
}
func (s *SearchCommand) SetFlags(fs *flag.FlagSet) {
	s.ScanCommand.setFlags(fs)
	fs.Var(&s.Expressions, "r", "`regex` to search for")
}

func (s *SearchCommand) Execute(ctx context.Context, fs *flag.FlagSet, args ...interface{}) subcommands.ExitStatus {
	s.Targets = fs.Args()
	if err := s.execute(ctx, fs, args...); err != nil {
		fmt.Printf("execute error: %s\n", err)
		return 1
	}

	return 0
}

func (s *SearchCommand) execute(ctx context.Context, f *flag.FlagSet, args ...interface{}) (err error) {
	s.ScanCommand.TestOnly = make(TestList, len(s.Expressions))

	for i, r := range s.Expressions {
		s.ScanCommand.TestOnly[i] = naiive.Test{
			Name:   r.String(),
			Regexp: r,
		}
	}

	return s.ScanCommand.execute(ctx, f, args...)
}
