#!/usr/local/bin/ruby

require 'aws-sdk-autoscaling'
require 'aws-sdk-elasticbeanstalk'

def main
  $stdout.sync = true # disable buffering output so output can be seen in real time and not delayed in Jenkins

  clusterID  = ENV["GRID_CLUSTER_ID"]
  branchName = ENV["GRID_BRANCH"]
  envName    = ENV["GRID_ENVIRONMENT"]
  hubENVName = ENV["GRID_HUB_ENVIRONMENT"]

  if clusterID.nil? || clusterID.empty? ||
    branchName.nil? || branchName.empty? ||
    envName.nil? || envName.empty? ||
    hubENVName.nil? || hubENVName.empty?
    raise 'You must provide environment variables GRID_CLUSTER_ID, GRID_BRANCH, GRID_ENVIRONMENT, GRID_HUB_ENVIRONMENT'
  end

  asgName   = "grid-win10-node-#{envName}-#{clusterID}"
  ebEnvName = "grid-hub-#{hubENVName}-#{clusterID}"

  client = Aws::AutoScaling::Client.new()
  ebClient = Aws::ElasticBeanstalk::Client.new()

  # Query the client for the Autoscaling Group
  asgResp = client.describe_auto_scaling_groups({
    auto_scaling_group_names: [ asgName, ],
  })

  # Ensure something is returned back, and it's only one
  if asgResp.auto_scaling_groups.length != 1
    raise "Problem finding asgName #{asgName}. Response: #{asgResp}"
  end

  # Ensure the ASG Name matches what the user requested
  if asgResp.auto_scaling_groups[0].auto_scaling_group_name != asgName
    raise "Expected #{asgName}, got: #{asgResp.auto_scaling_groups[0].auto_scaling_group_name}"
  end

  updateBranch(client, asgName, branchName)
  terminateNodes(client, asgResp.auto_scaling_groups[0].instances)

  # Restart the Hub Application Server
  sleep(10) # Ensure the instances have had a chance to start terminating
  puts "Restarting App Server for #{ebEnvName}"
  ebClient.restart_app_server({
    environment_name: ebEnvName,
  })
  sleep(60) # Wait for restart to complete

  # Wait for all the nodes in the ASG to register to the hub before returning
  waitForHubRegistration(ebClient, ebEnvName, asgResp.auto_scaling_groups[0].instances, clusterID)
end

# Updates the branch of an ASG
# @param [Aws::AutoScaling::Client] client The ASG Client
# @param [String] asgName The name of the autoscaling group
# @param [String] branchName the name of the branch to deploy
def updateBranch(client, asgName, branchName)
  puts "Updating Branch for ASG '#{asgName}' to '#{branchName}'"
  client.create_or_update_tags({
    tags: [
      {
        key: 'Branch',
        propagate_at_launch: true,
        resource_id: asgName,
        resource_type: 'auto-scaling-group',
        value: branchName,
      },
    ],
  })
end

# Terminate the nodes of a given ASG
# @param [Aws::AutoScaling::Client] client The ASG Client
# @param [Array] instances The instances to terminate (received from the ASG Response)
def terminateNodes(client, instances)
  count = 0
  instances.each do |instance|
    count += 1
    puts "[#{count}] Terminating #{instance.instance_id} with launch_configuration_name: #{instance.launch_configuration_name}"
    client.terminate_instance_in_auto_scaling_group({
      instance_id: instance.instance_id,
      should_decrement_desired_capacity: false,
    })
  end
end

# This method will wait for the Grid Nodes to Register to the Hub
# @param ebClient [Aws::ElasticBeanstalk::Client]
# @param ebEnvName [String] The eb environment to poll for registration status
# @param instances [Array] The instances that will register to the EB Environment
# @param clusterID The ID of the cluster name
def waitForHubRegistration(ebClient, ebEnvName, instances, clusterID)
  puts "Going to wait for nodes to register to the Hub"

  url = getRouterURL(ebClient, ebEnvName)
  uri = URI(url)

  startTime = Time.now
  timeoutMinutes = 45
  while Time.now < startTime + (60 * timeoutMinutes)
    puts "-> #{url}"
    resp = Net::HTTP.get_response(uri)
    puts "<- [#{resp.code}] #{url}"

    if resp.code != '200'
      puts "[WARN]: Received Response Code: #{resp.code} - Message: #{resp.message}"
      sleep(10)
      next
    end

    jsonResp = JSON.parse(resp.body)

    # Find the target hub from the Grid Router based on the cluster id
    # Wait for it to have all of the registered nodes
    hubResult = nil
    jsonResp.each do |r|
      if r.dig('clusterName') == clusterID && r.dig('healthy') == true
        hubResult = r
        break
      end
    end

    if hubResult == nil
      puts "[WARN]: Problem finding. Full Response: #{jsonResp}"
      sleep(10)
      next
    end

    totalSlots = hubResult.dig("SlotCounts", "Total")
    if totalSlots == nil
      puts "[WARN]: Could not find total slots. Full Response: #{jsonResp}"
      sleep(10)
      next
    end

    if totalSlots == instances.length
      puts "All #{instances.length} instances have registered to the hub"
      return
    else
      puts "Going to retry. Expected Registered Instances: #{instances.length}. Current Registered Instances: #{totalSlots}"
    end

    sleep(60)
  end

  # If it got here, there's a problem
  raise 'Timeout reached waiting for instances to register with hub'
end

# Gathers the Grid Router Registry URL from the Beanstalk Option Settings
# @param ebClient [Aws::ElasticBeanstalk::Client]
# @param ebEnvName [String] The eb environment to poll for registration status
def getRouterURL(ebClient, ebEnvName)
  router_url = nil
  configSettingsResp = ebClient.describe_configuration_settings(
      { application_name: "grid-hub", environment_name: ebEnvName, }
  )

  if configSettingsResp.configuration_settings.length != 1
    raise "Unexpected number of environments returned. Expected 1, Got: #{configSettingsResp.configuration_settings.length}.
      Output: #{configSettingsResp.to_h}"
  end

  configSettingsResp.configuration_settings[0].option_settings.each do |opt_setting|
    if opt_setting.option_name == "CBG_COMPANION_ROUTER_URL"
      router_url = opt_setting.value
      break
    end
  end

  if router_url.nil? || router_url == ""
    raise "Error fetching Router URL from Configuration Settings. Full Resp: #{configSettingsResp.to_h}"
  end

  return "#{router_url}/cbg/api/hub/registry"
end

main()
