#!/bin/bash
# This script will search all Browser Grid Accounts to see if they contain the given cluster
# Required Arguments: GRID_CLUSTER_ID, GRID_HUB_ENVIRONMENT
# Returns: Account # if Found (Status 0), Status 1 if not found

main() {
  export AWS_STS_REGIONAL_ENDPOINTS=regional

  if [ -z "$GRID_HUB_ENVIRONMENT" -o -z "$GRID_CLUSTER_ID" ]; then
      >&2 echo "Must provide \$GRID_HUB_ENVIRONMENT and \$GRID_CLUSTER_ID"
      exit 1
  fi

  # Grab all Grid AWS Accounts
  resetAWSCredentials
  getAccountManifest

  # Loop through each account in the manifest
  # Search for the grid hub environment & cluster in that account
  while read i; do
    # Assume the role for the account
    local accountID="$i"
    local assumeRoleARN="arn:aws:iam::$accountID:role/jenkins_grid_node_deploy"
    assumeRole "$assumeRoleARN"

    # Search the account for the grid hub cluster
    local environmentName="grid-hub-$GRID_HUB_ENVIRONMENT-$GRID_CLUSTER_ID"
    local result=$(aws elasticbeanstalk describe-environments --output json --environment-names "$environmentName")
    local resultCount=$(echo "$result" | jq '.[] | length')

    resetAWSCredentials

    if [ "$resultCount" -eq 1 ]; then
      echo "$accountID"
      rm accounts.json # clean up
      exit 0
    fi
  done < <(jq -r -c '.[].id' accounts.json)

  rm accounts.json # clean up
  >&2 echo "No match found"
  exit 1
}

# Assumes an AWS Role
# @param Account ID to assume
assumeRole() {
  local assumeRoleARN="$1"
  output=$(aws sts assume-role --role-arn "$assumeRoleARN" --role-session-name "Jenkins-Session" --duration-seconds 900)

  if [ $? -ne 0 ]; then
    >&2 echo "Error assuming role: $assumeRoleARN"
    exit 1
  fi

  # Set Credentials
  export AWS_ACCESS_KEY_ID=$(echo $output | jq -r '.Credentials.AccessKeyId')
  export AWS_SECRET_ACCESS_KEY=$(echo $output | jq -r '.Credentials.SecretAccessKey')
  export AWS_SESSION_TOKEN=$(echo $output | jq -r '.Credentials.SessionToken')
}

# Resets AWS Credentials, which helps between assuming roles to ensure your assumed-role isn't assuming another role
resetAWSCredentials() {
  unset AWS_ACCESS_KEY_ID
  unset AWS_SECRET_ACCESS_KEY
  unset AWS_SESSION_TOKEN
}

# Fetch the manifest of the account numbers for Browser Grid
# Downloads it to a file called accounts.json
getAccountManifest() {
  assumeRole "arn:aws:iam::147030575244:role/jenkins_grid_artifact_upload"

  aws s3 cp --quiet s3://grid-artifacts/accounts.json accounts.json
  local result="$?"

  resetAWSCredentials

  if [ "$result" -ne 0 ]; then
    >&2 echo "Error downloading account manifest"
    exit 1
  fi
}

main
