require 'aws-sdk-autoscaling'
require 'aws-sdk-ec2'
require 'aws-sdk-elasticbeanstalk'

def main
  $stdout.sync = true # disable buffering output so output can be seen in real time and not delayed in Jenkins

  clusterID    = ENV["GRID_CLUSTER_ID"]
  envName      = ENV["GRID_ENVIRONMENT"]
  instanceType = ENV["GRID_TARGET_INSTANCE_TYPE"]
  hubENVName   = ENV["GRID_HUB_ENVIRONMENT"]

  if clusterID.nil? || clusterID.empty? ||
      instanceType.nil? || instanceType.empty? ||
      envName.nil? || envName.empty? ||
      hubENVName.nil? || hubENVName.empty?
    raise 'You must provide environment variables GRID_CLUSTER_ID, GRID_ENVIRONMENT, GRID_TARGET_INSTANCE_TYPE, GRID_HUB_ENVIRONMENT'
  end

  asgName = "grid-win10-node-#{envName}-#{clusterID}"
  ebEnvName = "grid-hub-#{hubENVName}-#{clusterID}"

  # Create Clients to interact with the sdk
  asgClient = Aws::AutoScaling::Client.new(profile: 'twitch-cape-qe-aws')
  ec2Client = Aws::EC2::Client.new(profile: 'twitch-cape-qe-aws')
  ebClient  = Aws::ElasticBeanstalk::Client.new()

  # Query the client for the Autoscaling Group
  asgResp = asgClient.describe_auto_scaling_groups({ auto_scaling_group_names: [ asgName, ], })

  # Ensure something is returned back, and it's only one
  if asgResp.auto_scaling_groups.length != 1
    raise "Problem finding asgName #{asgName}. Response: #{asgResp}"
  end

  # Fetch the Target ASG from the Response
  targetASG = asgResp.auto_scaling_groups[0]

  # Ensure the ASG Name matches what the user requested
  if targetASG.auto_scaling_group_name != asgName
    raise "Expected #{asgName}, got: #{targetASG.auto_scaling_group_name}"
  end

  # Stop the instances so that the instance type can be modified
  stopInstances(ec2Client, targetASG.instances)

  # Modify the instance type
  modifyInstanceType(ec2Client, targetASG.instances, instanceType)

  # Restart the Hub Application Server
  puts "Restarting App Server for #{ebEnvName}"
  ebClient.restart_app_server({
    environment_name: ebEnvName,
  })

  # Start the instances again
  startInstances(ec2Client, targetASG.instances)
end

# @param ec2Client Aws::EC2::Client The ec2 client to use for making requests
# @param instances Array<Types::Instance> The instances to stop
def stopInstances(ec2Client, instances)
  instanceIDs = getInstanceIDsFromASGResp(instances)

  puts "Making stop instance request to: #{instanceIDs}"

  ec2Client.stop_instances({
    instance_ids: instanceIDs, # required
    hibernate: false,
    dry_run: false,
    force: false,
  })

  begin
    puts 'Waiting for instances to stop...'
    ec2Client.wait_until(:instance_stopped, instance_ids: instanceIDs)
    puts 'Instances successfully stopped'
  rescue Aws::Waiters::Errors::WaiterFailed => error
    puts "Failed waiting for instance to stop: #{error.message}"
    raise error
  end
end

# @param ec2Client Aws::EC2::Client The ec2 client to use for making requests
# @param instances Array<Types::Instance> The instances to stop
def startInstances(ec2Client, instances)
  instanceIDs = getInstanceIDsFromASGResp(instances)

  puts "Making start instance request to: #{instanceIDs}"

  ec2Client.start_instances({
    instance_ids: instanceIDs,
  })

  begin
    puts 'Waiting for instances to start...'
    ec2Client.wait_until(:instance_running, instance_ids: instanceIDs)
    puts 'Instances successfully started'
  rescue Aws::Waiters::Errors::WaiterFailed => error
    puts "failed waiting for instance to start: #{error.message}"
    raise error
  end
end

# @param ec2Client Aws::EC2::Client The ec2 client to use for making requests
# @param instances Array<Types::Instance> The instances to stop
# @param targetInstanceType String The target instance type to change the instances to
def modifyInstanceType(ec2Client, instances, targetInstanceType)
  count = 0
  instances.each do |instance|
    puts "[#{count}] Updating #{instance.instance_id} to Instance Type: #{targetInstanceType}"
    resp = ec2Client.modify_instance_attribute({
      instance_id: instance.instance_id,
      instance_type: {
        value: targetInstanceType,
      },
    })
  puts resp
  end
end

# @param instances Array<Types::Instance> The instances to get the ids from
# @return Array<String> A string of instance ids
def getInstanceIDsFromASGResp(instances)
  instanceIDs = []
  instances.each do |instance|
    instanceIDs << instance.instance_id
  end
  return instanceIDs
end

main
