#!/bin/bash

set -e # exit when any command fails

main() {
  # Store path of this script so paths can be made towards it
  parentPath=$( cd "$(dirname "${BASH_SOURCE[0]}")" ; pwd -P )

  echo "Finding Account ID for Cluster"
  accountID=$(eval "$parentPath/find_account_of_cluster.sh")

  if [ "$?" -ne 0 -o -z "$accountID" ]; then
    >&2 echo "Error finding account id"
    exit 1
  fi

  echo "Account ID containing cluster: $accountID"

  assumeRoleARN="arn:aws:iam::$accountID:role/jenkins_grid_node_deploy"
  echo "Assuming Role: $assumeRoleARN"
  assumeRole "$assumeRoleARN"

  # If a chef deployment, run the deployChef() function
  if [ "$DEPLOY_TYPE" == "chef" ]; then
      deployChef
  elif [ "$DEPLOY_TYPE" == "modify_instance_type" ]; then
      docker build -t grid-node-deploy "$parentPath"
      modifyInstanceType
  else
      docker build -t grid-node-deploy "$parentPath"
      deployReplace
  fi

  resetAWSCredentials
}

# Assumes an AWS Role
# @param Account ID to assume
assumeRole() {
  local assumeRoleARN="$1"
  output=$(aws sts assume-role --role-arn "$assumeRoleARN" --role-session-name "Jenkins-Session" --duration-seconds 900)

  if [ $? -ne 0 ]; then
    >&2 echo "Error assuming role: $assumeRoleARN"
    exit 1
  fi

  # Set Credentials
  export AWS_ACCESS_KEY_ID=$(echo $output | jq -r '.Credentials.AccessKeyId')
  export AWS_SECRET_ACCESS_KEY=$(echo $output | jq -r '.Credentials.SecretAccessKey')
  export AWS_SESSION_TOKEN=$(echo $output | jq -r '.Credentials.SessionToken')
}

# Resets AWS Credentials, which helps between assuming roles to ensure your assumed-role isn't assuming another role
resetAWSCredentials() {
  unset AWS_ACCESS_KEY_ID
  unset AWS_SECRET_ACCESS_KEY
  unset AWS_SESSION_TOKEN
}

# Runs a Chef Deployment
deployChef() {
  echo "Preparing Chef Deployment"

  TARGETS=$(cat <<-END
      [
        {"Key":"tag:GridResourceType","Values":["node"]},
        {"Key":"tag:GridClusterID","Values":["$GRID_CLUSTER_ID"]},
        {"Key":"tag:Environment","Values":["$GRID_ENVIRONMENT"]},
        {"Key":"tag:Service","Values":["grid"]}
      ]
END
)

  # Trigger an SSM Command to update and run chef
  commandID=$(aws ssm send-command --document-name "Update_And_Run_Chef_Solo" \
    --document-version "\$LATEST" \
    --targets "$TARGETS" \
    --parameters "{\"branch\":[\"$GRID_BRANCH\"]}" \
    --timeout-seconds 900 \
    --max-concurrency "100%" \
    --max-errors "0"\
    --output text \
    --query 'Command.CommandId'
  )

  echo "The command id: $commandID"

  # Print out detail link
  isengardLink="https://isengard.amazon.com/federate?account=$accountID&role=Admin&destination=systems-manager%2Frun-command%2F$commandID%3Fregion%3D$AWS_DEFAULT_REGION"
  echo -e "Isengard Link to Command (WARNING - This will sign you into the account):\n$isengardLink"
  sleep 5

  # Wait for the command to no longer be in progress
  status="InProgress"
  while [ "$status" == "InProgress" ]
  do
    result=$(aws ssm list-commands --command-id "$commandID" \
      --output json
    )
    status=$(echo $result |         jq -r '.Commands[0].Status')
    targetCount=$(echo $result |    jq -r '.Commands[0].TargetCount')
    completedCount=$(echo $result | jq -r '.Commands[0].CompletedCount')
    echo "Current status: $status - $completedCount/$targetCount"
    sleep 5
  done

  echo -e "Command Complete. View detail (WARNING - This will sign you into the account):\n$isengardLink"

  # Set the build status properly depending on if it's a success or not
  if [ "$status" != "Success" ]; then
    exit 1
  fi
}

deployReplace() {
  echo "Preparing Full Replace Deployment"
  docker run --rm \
    -v "$parentPath":/usr/src/myapp \
    -w /usr/src/myapp \
    -e GRID_CLUSTER_ID="$GRID_CLUSTER_ID" \
    -e GRID_ENVIRONMENT="$GRID_ENVIRONMENT" \
    -e GRID_HUB_ENVIRONMENT="$GRID_HUB_ENVIRONMENT" \
    -e GRID_BRANCH="$GRID_BRANCH" \
    -e AWS_REGION="$AWS_DEFAULT_REGION" \
    -e AWS_ACCESS_KEY_ID="$AWS_ACCESS_KEY_ID" \
    -e AWS_SECRET_ACCESS_KEY="$AWS_SECRET_ACCESS_KEY" \
    -e AWS_SESSION_TOKEN="$AWS_SESSION_TOKEN" \
    grid-node-deploy ruby deploy.rb
}

modifyInstanceType() {
  echo "Preparing to Modify Instance Type"
  docker run --rm \
    -v "$parentPath":/usr/src/myapp \
    -w /usr/src/myapp \
    -e GRID_CLUSTER_ID=$GRID_CLUSTER_ID \
    -e GRID_ENVIRONMENT=$GRID_ENVIRONMENT \
    -e GRID_HUB_ENVIRONMENT=$GRID_HUB_ENVIRONMENT \
    -e GRID_TARGET_INSTANCE_TYPE="$GRID_TARGET_INSTANCE_TYPE" \
    -e AWS_REGION="$AWS_DEFAULT_REGION" \
    -e AWS_ACCESS_KEY_ID="$AWS_ACCESS_KEY_ID" \
    -e AWS_SECRET_ACCESS_KEY="$AWS_SECRET_ACCESS_KEY" \
    -e AWS_SESSION_TOKEN="$AWS_SESSION_TOKEN" \
    grid-node-deploy ruby modify_instance_type.rb
}

main
