﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Runtime.InteropServices;
using WUApiLib;
using Microsoft.Win32;
using System.IO;
using System.Threading;

// Contains script for downloading and installing Windows Updates for Win10.
namespace WindowsTools
{
    /// <summary>
    /// Application for running Windows Updates and interacting with the WUApi
    /// </summary>
    public static class WindowsUpdate
    {
        /// <summary>
        /// Searches for available Windows Updates
        /// </summary>
        /// <returns>A collection of available updates</returns>
        public static IUpdateCollection Search()
        {
            // Search for Visible Software that is not yet installed
            String searchConditions = "IsInstalled=0 and Type='Software' and IsHidden=0";
            IUpdateSession session; // The Windows Update session
            IUpdateSearcher searcher; // Object to search for updates
            ISearchResult searchResults; // Results from the search
            IUpdateCollection availabeUpdates; // Search available updates
            IUpdateCollection updatesToDownload; // The updates that are eligible to download (i.e., no user inptu)
            IUpdateServiceManager2 manager; // Object for the Update Manager
            IUpdateServiceCollection servicesCol; // Store the manager's services

            // Opt In to Microsoft Update
            // https://msdn.microsoft.com/en-us/library/windows/desktop/aa826676(v=vs.85).aspx
            manager = new UpdateServiceManager();
            manager.AddService2("7971f918-a847-4430-9279-4a52d1efe18d", 7, "");
            manager.ClientApplicationID = "Twitch Grid";

            // Print out some information concerning the configured managers
            Logger.Debug("Manager Service Info ");
            servicesCol = manager.Services;
            foreach (IUpdateService service in servicesCol)
            {
                Logger.Debug("Service Name: " + service.Name + " Service ID: " + service.ServiceID);
            }

            // Start the session
            session = new UpdateSession();

            // Create a searcher and find the available updates
            try {
                searcher = session.CreateUpdateSearcher();
                searcher.ServerSelection = ServerSelection.ssOthers;
                searcher.ServiceID = "7971f918-a847-4430-9279-4a52d1efe18d";

                Logger.Info("Searching for Windows Updates...");
                searchResults = searcher.Search(searchConditions);
                availabeUpdates = searchResults.Updates; // Grab the updates from the collection
            } catch (Exception e) {
                Console.WriteLine("Encountered exception: " + e.ToString() + " Stack: " + e.StackTrace.ToString());
                throw e;
            }

            // Loop through all availableUpdates to determine if they can be downloaded
            updatesToDownload = new UpdateCollection();
            foreach (IUpdate update in availabeUpdates)
            {
                Boolean installThisUpdate = false; // Default to false. Keep track if it should use to update

                Logger.Debug("Inspecting Update " + update.Title);

                // Check if the update requests user input
                Logger.Debug("Update CanRequestUserInput: " + update.InstallationBehavior.CanRequestUserInput);
                if (update.InstallationBehavior.CanRequestUserInput)
                {
                    Logger.Warn("This update requested user input. Skipping...");
                    continue;
                }
                else // No user input, can install the update
                {
                    installThisUpdate = true;

                    // Check if there is a Eula to accept, and auto accept it
                    Logger.Debug("Update EulaAccepted: " + update.EulaAccepted);
                    if (!update.EulaAccepted)
                    {
                        Logger.Info("Accepting EULA for " + update.Title);
                        update.AcceptEula();
                    }
                }

                if (installThisUpdate)
                {
                    Logger.Info("Available Update: " + update.Title);
                    updatesToDownload.Add(update); // Add it to the updatesToDownload collection
                }
            }

            if (updatesToDownload.Count == 0)
            {
                Logger.Info("No updates to download");
            }

            Logger.Info("Found " + updatesToDownload.Count + " updates");
            return updatesToDownload; // Return the collection of updates to download
        }

        /// <summary>
        /// Download the Windows Updates to the Disk
        /// </summary>
        /// <returns>The collection of the updates indicated to download</returns>
        /// <param name="updatesToDownload">Collection of the updates to download</param>
        public static IUpdateCollection Download(IUpdateCollection updatesToDownload)
        {
            IUpdateDownloader downloader; // Object to download updates
            IDownloadResult downloadResult; // Results of the download

            downloader = new UpdateDownloader();

            // Set the updates the downloader should update
            // Casts to UpdateCollection, due to issues using IUpdateCollection with non IUpdateDownloader
            downloader.Updates = (WUApiLib.UpdateCollection)updatesToDownload;

            try
            {
                Logger.Info("Downloading Updates...");
                downloadResult = downloader.Download(); // Start downloading updates
            }
            catch (COMException e)
            {
                if ((uint)e.ErrorCode == 0x80240044) // Likely doesn't have admin permissions
                {
                    // https://social.msdn.microsoft.com/Forums/vstudio/en-US/1332db4b-0c80-4d38-aef9-57735122421f/windows-update-agent-in-windows-7?forum=clr
                    Logger.Error("Encountered Exception: " + e + "\n\nTry running with Administrator Privileges.");
                }

                throw; // Re-throw
            }

            // Take a look at the download code
            Logger.Debug("downloadResult ResultCode: " + downloadResult.ResultCode);
            if (downloadResult.ResultCode == OperationResultCode.orcSucceeded)
            {
                Logger.Info("Updates downloaded successfully");
            }
            else
            {
                Logger.Error("Unexpected result code in downloaded updates: " + downloadResult.ResultCode);
            }

            return updatesToDownload;
        }

        /// <summary>
        /// Loops through the Updates to Install, making sure the update was successfully downloaded
        /// </summary>
        /// <returns>The updates to install.</returns>
        /// <param name="updates">Collection of downloaded updates.</param>
        public static IUpdateCollection compileUpdatesToInstall(IUpdateCollection updates)
        {
            IUpdateCollection updatesToInstall = new UpdateCollection();

            foreach (IUpdate update in updates)
            {
                Logger.Debug("Analyzing download results for " + update.Title);

                Logger.Debug("Update IsDownloaded: " + update.IsDownloaded);
                if (update.IsDownloaded)
                {
                    Logger.Debug("Adding to updatesToInstall");
                    updatesToInstall.Add(update);
                }
            }

            return updatesToInstall;
        }

        /// <summary>
        /// Install the specified updatesToInstall.
        /// </summary>
        /// <returns>The collection of updates provided to install.</returns>
        /// <param name="updatesToInstall">Updates to install.</param>
        public static void Install(IUpdateCollection updatesToInstall)
        {
            IUpdateSession session;
            IUpdateInstaller installer; // The installer object
            IInstallationResult result; // The result of the installation

            Logger.Info("Installing " + updatesToInstall.Count + " updates");
            session = new UpdateSession();

            // Create the update installer, set the updates to install
            installer = session.CreateUpdateInstaller();
            installer.Updates = (WUApiLib.UpdateCollection)updatesToInstall;

            // Install updates
            try {
                result = installer.Install();
            }
            catch (Exception e) {
                Logger.Error("Upon installing, encountered exception: " + e.Message.ToString() + " Stack: " + e.StackTrace.ToString());
                throw;
            }

            Logger.Info("Installation result: " + result);
            PrintInstallSummary(updatesToInstall, result);
        }

        /// <summary>
        /// Prints a summary of the updates that were installed
        /// </summary>
        /// <param name="updates">The updates that were specified to install.</param>
        /// <param name="result">The result returned from the install object.</param>
        public static void PrintInstallSummary(IUpdateCollection updates, IInstallationResult result)
        {
            Logger.Info("Summary of installations:");

            // Loop through all updates, printing its title and result
            for (int i = 0; i < updates.Count; i++)
            {
                Logger.Info(i + " " + updates[i].Title + ": Result: " + result.GetUpdateResult(i).ResultCode);
            }
        }

        /// <summary>
        /// Restart this instance.
        /// </summary>
        private static void Restart()
        {
            Logger.Info("Rebooting Computer...");
            System.Diagnostics.Process.Start("shutdown.exe", "-r -t 5 -c \"Restarting computer for Windows Updates\"");
        }

        /// <summary>
        /// The entry point of the program, where the program control starts and ends.
        /// </summary>
        /// <returns>
        /// The exit code that is given to the operating system after the program ends.
        /// <c>0</c> if more updates may need to be installed
        /// <c>2</c> if all updates are complete, and the script can deregister itself
        /// </returns>
        public static int Main()
        {
            IUpdateCollection availableUpdates, downloadedUpdates, updatesToInstall;

            // Search for available updates
            availableUpdates = Search();

            // If there are no updates, return
            if (availableUpdates.Count == 0)
            {
                Logger.Debug("No more updates are available, returning.");
                return 2;
            }

            downloadedUpdates = Download(availableUpdates);
            updatesToInstall = compileUpdatesToInstall(downloadedUpdates);
            Install(updatesToInstall);
            Restart(); // Restart to allow updates to install

            return 0;
        }
    }

    /// <summary>
    /// Logger class to provide ability to log messages to a file and console output.
    /// </summary>
    public static class Logger
    {
        public static void Debug(String msg)
        {
            String prefix = "[DEBUG]";
            log(prefix, msg);
        }

        public static void Info(String msg)
        {
            String prefix = "[INFO]";
            log(prefix, msg);
        }

        public static void Warn(String msg)
        {
            string prefix = "[WARN]";
            log(prefix, msg);
        }

        public static void Error(String msg)
        {
            string prefix = "[ERROR]";
            log(prefix, msg);
        }

        /// <summary>
        /// Log the specified prefix and msg to stdout and a file
        /// </summary>
        /// <param name="prefix">The type of message, i.e., info</param>
        /// <param name="msg">The message to log</param>
        private static void log(String prefix, String msg)
        {
            String message = prefix + " " + msg;
            Console.WriteLine(message);
            WriteToFile(message);
        }

        /// <summary>
        /// Writes a message to a file
        /// </summary>
        /// <param name="message">The message to log</param>
        private static void WriteToFile(String message)
        {
            String logMessage;
            String log_file = "C:\\Windows\\Temp\\win_update_log.txt";

            logMessage = DateTime.Now.ToString() + ": " + message + Environment.NewLine;

            // Try to write to file, with max attempts of 10
            for (int attempts = 0; attempts < 10; attempts++)
            {
                try
                {
                    File.AppendAllText(log_file, logMessage);
                    break; // It was successful, break out of the for loop
                }
                catch (IOException)
                {
                    Console.WriteLine("Warning: Caught an IO Exception");
                    Thread.Sleep(1000);
                }
            }
        }
    }
}
