#
# Cookbook:: grid_node
# Recipe:: selenium
#

desiredVersion   = node['cbg']['version']
cdStoragePath    = node['cbg']['path']
installedVersion = cbg_version()
versionChange    = !installedVersion.nil? && installedVersion != desiredVersion # if installed, and versions changing

Chef::Log.info("Configuring Selenium CBG version #{desiredVersion}")

# To be able to close java
chef_gem 'sys-proctable' do
  action :install
end

ruby_block 'close_java' do
  action :run
  block do
    close_java()
  end
  only_if { versionChange }
  notifies :run, 'windows_task[relaunch_selenium]', :delayed
end

# Remove existing only if we are changing the version
directory cdStoragePath do
  action :delete
  only_if { versionChange }
  recursive true
  notifies :run, 'ruby_block[close_java]', :before
end

aws_s3_file "#{cdStoragePath}.zip" do
  action :create
  bucket 'grid-artifacts'
  remote_path "cbg_bin/#{desiredVersion}.zip"
  region 'us-west-2'
end

# rubocop:disable ChefModernize/PowershellScriptExpandArchive
powershell_script 'extract' do
  action :run
  code "Expand-Archive #{cdStoragePath}.zip -DestinationPath #{node['config_directory']}"
  only_if { installedVersion.nil? || versionChange }
end

template "#{cdStoragePath}\\cbg_config.yaml" do
  source 'selenium/cbg_config.erb'
  mode  '0666'
  owner 'selenium'
  group 'Administrators'
  variables(instanceID: node['ec2']['instance_id'])
end

# Create a VERSION file to help determine what version is currently installed
file "#{cdStoragePath}/VERSION.txt" do
  content desiredVersion
  mode  '0666'
  owner 'selenium'
  group 'Administrators'
end

# Register job to start up if it's the first time installing
windows_task 'selenium_startup' do
  action :create
  command "powershell -File #{node['config_directory']}\\launch.ps1"
  description 'Launch Selenium on Windows Startup'
  frequency :on_logon
  user      'selenium'
  password  windowsPassword()
  sensitive true
end

# Relaunch Selenium if there was a version change (and this isn't the first install)
windows_task 'relaunch_selenium' do
  action :create
  command "powershell -File #{node['config_directory']}\\launch.ps1"
  description 'Launch Selenium'
  start_when_available true
  frequency :none
  user      'selenium'
  password  windowsPassword()
  sensitive true
end
