# This script configures Internet Explorer & IE Webdriver
# Registry Key entry reference:
# https://support.microsoft.com/en-us/help/182569/internet-explorer-security-zones-registry-entries-for-advanced-users

$config_directory = "C:\ProgramData\grid-config"

function Main {
    Write-Log -message "Provisioning Internet Explorer" -level "INFO"
    Download("3.150.0")
    Configure-IE
}

# Downloads IE Driver specified by version
# @param [String] version The version of IE Driver Server to download
# @return [String] The path to the folder where the driver
function Download($version) {
    # Build URLs and Paths
    $folderPath = "$config_directory\iedriver"
    $path = "$folderPath.zip"
    Write-Log -message "Downloading IEDriverServer to $path" -level "DEBUG"
    try {
        Read-S3ObjectWithWait -s3Uri "s3://grid-artifacts/ie/IEDriverServer_x64_$($version).zip" -localPath "$path"
    } catch {
        Write-Log -level "ERROR" -message "While downloading IE, encountered exception: $($_.Exception.Message)"
        throw $_
    }

    # Make sure the file exists
    if (!(Test-Path $path))
    {
        $errMsg = "After download, could not find path $path"
        Write-Log -level "ERROR" -message $errMsg
        throw $errMsg
    }

    # Unzip
    Expand-Archive $path -DestinationPath $folderPath
    return $folderPath
}

# Configures Internet Explorer Driver to be able to run
# Reference: https://github.com/SeleniumHQ/selenium/wiki/InternetExplorerDriver#required-configuration
function Configure-IE() {
    Enable-PersistentConnection
    Disable-ProtectedMode
    Disable-FirstRunCustomize
}

# Allows the ie driver to maintain a connection to IE
function Enable-PersistentConnection() {
    $key = 'HKLM:\SOFTWARE\Wow6432Node\Microsoft\Internet Explorer\Main\FeatureControl\FEATURE_BFCACHE'
    New-Item -Force -Path $key
    New-ItemProperty -Force -Path $key -Name "iexplore.exe" -PropertyType DWord -Value 0
}

# Disables protected mode across all zones
# "On IE 7 or higher on Windows Vista or Windows 7, you must set the Protected Mode settings for each zone to be the same value."
function Disable-ProtectedMode() {
    # We need to set the registry for the Selenium User - not the current user because this script runs as root
    $userID = Get-UserID("selenium")

    if ([string]::IsNullOrEmpty($userID)) {
        $errMsg = "Unable to find user id for Selenium when setting up Internet Explorer"
        Write-Log -level "ERROR" -message $errMsg
        throw $errMsg
    }

    # Load the HKEY_USERS registry as HKU, and fetch the user id for the Selenium username
    New-PSDrive -PSProvider Registry -Name HKU -Root HKEY_USERS

    # Make a key that points to the Selenium user account
    $baseKey = "HKU:\$userID\Software\Microsoft\Windows\CurrentVersion\Internet Settings\Zones"

    # Set the same for zones 0-4
    For ($zone=0; $zone -le 4; $zone++) {
        # the key will be \Zones\0 through 4
        $key = "$baseKey\$zone"

        # Value: 3 = Disable, 0 = Enable
        New-ItemProperty -Force -Path $key -Name "2500" -PropertyType DWord -Value 3
        Write-Log -level "DEBUG" -message "Wrote IE Reg Key: $key"
    }
}

# Disables the initial first time prompt that appears to set up IE
function Disable-FirstRunCustomize() {
    $key = "HKLM:\Software\Policies\Microsoft\Internet Explorer\Main"
    New-Item -Force -Path $key
    New-ItemProperty -Force -Path $key -Name "DisableFirstRunCustomize" -PropertyType DWord -Value 1
}

# Fetches the registry user id of the requested username
function Get-UserID($username) {
    # All profiles are listed under the ProfileList, grab their names
    $results = Get-Item "HKLM:\SOFTWARE\Microsoft\Windows NT\CurrentVersion\ProfileList\*"

    # Loop through each, grab just their child name (without the base)
    foreach ($profile in $results) {
        $id = $profile.PSChildName

        # See if the Image Path property value is equal to the C:\Users\USERNAME that we're looking for
        $imagePath = Get-ItemPropertyValue "HKLM:\SOFTWARE\Microsoft\Windows NT\CurrentVersion\ProfileList\$id" -Name "ProfileImagePath"
        if ($imagePath -eq "C:\Users\$username") {
            return $id
        }
    }
}

Main
