# This script is responsible for running all tasks that need to be done at startup
# Set instance tag "AutoLaunchGrid" to false to not run this.
# This currently involves
#   - Setting Resolution
#   - Starting Grid Cmd
param(
  [string]$seVersion
)

$ErrorActionPreference = "Stop" # Stop on any exception

# Import Modules
Import-Module "CBGLogger"
Import-Module "CBGAws"

$config_directory = "C:\ProgramData\grid-config"
$logDirectory = "C:\Users\selenium\AppData\Roaming\Grid"
$logFile = "launcher.log"
$logPath = "$logDirectory\$logFile"

# This method will look to see if the AWS Instance has a tag "AutoLaunchGrid"
#   and if it's false
# @return [Boolean] Whether the grid should launch or not
function shouldLaunch() {
  $result = Get-AWS-Tag("autolaunchgrid")

  if($result -eq "true") {
    return $true
  } else { # Generally the string "false" or $null
    return $false
  }
}

###############################
# Check if Grid should launch #
###############################

Write-Log -message "Running Launch Scripts" -level "INFO"

# Looks to see if instance has a tag "AutoLaunchGrid" set to false
$respShouldLaunch = shouldLaunch($tags) # Get a response if grid should launch
Write-Log -message "The tag AuthLaunchGrid is set to: $respShouldLaunch" -level "DEBUG"

if(!$respShouldLaunch)
{
  Write-Log -message "The instance had the tag AutoLaunchGrid set to false. Exiting." -level "WARN"
  exit 0
}

################
# Other Checks #
################

# Get the Hub's FQDN
$hubFQDN = Get-AWS-Tag("Hub_FQDN")
if(!$hubFQDN) {
  $errMsg = "Could not determine the instance's Hub FQDN. Ensure tags.Hub_FQDN is set."
  Write-Log -level "ERROR" -message $errMsg
  throw $errMsg
}

##################
# Set Resolution #
##################

Write-Log -message "Setting Resolution" -level "INFO"

# Set the Width and Height (width x height)
$width = "1280"
$height = "1024"
# Note: Argument list has `" because the App-Launcher needs to get an argument with spaces, so it gets escaped
& "$config_directory\app_launcher.exe" "`"C:\\ProgramData\\grid-config\\ResolutionConsole.exe set -width $width -height $height`"" 2>&1 | Out-File "$logPath" -Append -Encoding "UTF8"
Write-Log -message "Resolution Set" -level "DEBUG"

##############
# Start Grid #
##############

Write-Log -message "Launching Grid" -level "INFO"

# Launch either Selenium 3 or 4 depending on the SeMajorVersion tag
$seMajorVersion = Get-AWS-Tag("SeMajorVersion")
if($seMajorVersion -eq "4") {
  # Make sure user provided required Selenium 4 param
  if ([string]::IsNullOrWhiteSpace($seVersion)) {
    $msg = "Must provide SeVersion as a parameter. Example: 4.0.0-beta-1"
    Write-Log -level "ERROR" -message $msg
    throw $msg
  }

  $launchFilePath = "$config_directory\launch_selenium_4.ps1"
} else {
  $launchFilePath = "$config_directory\start_grid.ps1"
}

# Note: Argument list has `" because the App-Launcher needs to get an argument with spaces, so it gets escaped
& "$config_directory\app_launcher.exe" "`"powershell.exe -NoExit -ExecutionPolicy Bypass -file $launchFilePath $hubFQDN $seVersion`"" 2>&1 | Out-File "$logPath" -Append -Encoding "UTF8"
Write-Log -message "Grid Launched" -level "INFO"
