<#
This Script is responsible for provisioning Windows 10 Instances.
It goes through the steps of:
Downloading & Installing Chrome (& chromedriver)
Downloading Selenium Server
Downloading and installing JDK
Downloading and running files to launch Grid
#>

param(
  [string]$grid_environment,
  [string]$s3_url
)

# Main directory for storing Grid files
$config_directory = "C:\ProgramData\grid-config"

# Directory for logs
$logDirectory = "C:\Users\selenium\AppData\Local\Twitch\Grid\Logs"
New-Item -type directory $logDirectory

# Download & Configure Modules
& "$config_directory\configure_modules.ps1" $grid_environment

Write-Log -message "Beginning Provisioning Process. See DEBUG level for more detail." -level "INFO"

Write-Log -message "Reading from S3 URL: $s3_url" -level "DEBUG"
Write-Log -message "Grid environment: $grid_environment" -level "DEBUG"

# Activate Windows
& "$config_directory\activate_windows.ps1"

# Download AWS CLI
& "$config_directory\install_aws_cli.ps1"

# Monitor Processes
& "$config_directory\install_sysmon.ps1"

#######################
# Download Main Files #
#######################

& "$config_directory\install_sandstorm.ps1"

# Run Chef
& "$config_directory\run_chef.ps1"

# Configure Microsoft Edge
& "$config_directory\configure_edge.ps1"

# Configure Internet Explorer
& "$config_directory\configure_ie.ps1"

# Create the Selenium Configuration
& "$config_directory\build_node_config.ps1"

# Download supporting tools
& "$config_directory\get_supporting_tools.ps1" -gridEnvironment $grid_environment

# Install Nginx
& "$config_directory\install_nginx.ps1"

#######################################
# Local Security Policy configuration #
#######################################

# In Local Policies -> User Rights Assignment -> Replace a Process Level Token, Administrator needs to be added
# If not, when running CreateProcessAsUser, 1314 Error Code will be returned as it doesn't have permission
# Uses secedit. https://docs.microsoft.com/en-us/windows/device-security/security-policy-settings/administer-security-policy-settings#a-href-idbkmk-seceditausing-the-secedit-command-line-tool
Write-Log -message "Setting security permissions" -level "INFO"

$policy_file = "c:\secpol.cfg"
secedit /export /cfg $policy_file # Export the current policy
# Grab that policy, replace the value for SeAssignPrimaryTokenPrivilege to trust the Selenium account. Output it as same file
(gc $policy_file) -replace "SeAssignPrimaryTokenPrivilege = .*$", "SeAssignPrimaryTokenPrivilege = *S-1-5-19,*S-1-5-20,selenium" | Out-File $policy_file
# Load that config file into the security policy database
# For some reason /db C:\Windows\security\database\secedit.sdb wasn't working.
# Need to use /quiet or otherwise a prompt will appear
secedit /configure /db secedit.sdb /cfg $policy_file /overwrite /quiet
Remove-Item $policy_file -force

# Register Crons
& "$config_directory\configure_crons.ps1"

#############
# Finishing #
#############

# Restarts the computer. Currently this is needed for the security policy to take effect
Write-Log -message "Provisioning complete. Restarting computer" -level "INFO"
Restart-Computer -force
