# Contains AWS Utilities

# @return [String] The instance ID
Function Get-InstanceId {
    $config_directory = "C:\ProgramData\grid-config"
    $moduleDirectory = "$($config_directory)\modules"
    $moduleName = "CBGModules.dll"

    Add-Type -Path "$moduleDirectory\$moduleName"
    return [CBGModules.Aws]::GetInstanceID()
}

# @return [Array] Tags of the instance (in a key/value json format)
Function Get-AWS-Tags() {
    param (
        [int]$maxAttempts = 10,
        [int]$attemptDelay = 1 # The number of seconds to delay an attempt
    )

    # Sometimes AWS can return exit code 255, so let's retry if the exit code isn't 0
    $retryCount=0
    Do {
        $retryCount++
        $instanceID = Get-InstanceId
        $instanceData = & aws ec2 describe-instances --region us-west-2 --instance-ids $instanceID

        if($LastExitCode -ne 0) {
            $msg = "[$retryCount/$maxAttempts] When describing instances, the AWS CLI returned an exit code of $LastExitCode"

            if($retryCount -eq $maxAttempts -or $retryCount -gt $maxAttempts) {
                Write-Log -message $msg -level "ERROR"
                Write-Error $msg -ErrorAction Stop # Break the loop, max retries reached
            } else { # Retry with a sleep...
                Write-Log -message $msg -level "INFO"
                Start-Sleep -s $attemptDelay
            }
        }
    } While ($LastExitCode -ne 0)

    $jData = $instanceData | ConvertFrom-Json # Transfer the returned data to a json structure
    return $jData.Reservations.Instances.Tags
}

# @return [String] The value of the tag. Returns null if the tag is not included
Function Get-AWS-Tag() {
    param([string]$tagName)
    $instanceID = Get-InstanceID
    $tags = Get-AWS-Tags

    for ($i=0; $i -lt $tags.length; $i++) {
        if ($tags[$i].Key.ToLower() -eq $tagName) # If the tag exists...
        {
            return $tags[$i].Value
        }
    }

    return $null
}

# @return [String] The region the AWS Instance is running in.
Function Get-AWS-Region() {
    $awsInstanceData = Invoke-RestMethod -Uri http://169.254.169.254/latest/dynamic/instance-identity/document
    $region = $awsInstanceData.region
    if ([string]::IsNullOrEmpty($region)) { # Ensure it's not blank
        throw "Region was null or empty within the instance data"
    }

    return $awsInstanceData.region
}

# Downloads an object from S3, waiting to complete before returning
# @param $s3Uri The s3 uri to download from. Example: s3://grid-artifacts/my-object
# @param $localPath The path to download to
Function Read-S3ObjectWithWait() {
    param(
        [string]$s3Uri,
        [string]$localPath
    )

    # Need to use Start-Process, for some reason Powershell Read-S3Object and `& aws s3 cp` doesn't wait for completion
    $proc = Start-Process -FilePath "aws" -ArgumentList "s3 cp `"$s3Uri`" `"$localPath`"" -Wait -NoNewWindow -PassThru
    if ($proc.ExitCode -ne 0) {
        throw "error downloading $s3Uri"
    }
}

export-modulemember -function Get-InstanceId
export-modulemember -function Get-AWS-Tags
export-modulemember -function Get-AWS-Tag
export-modulemember -function Get-AWS-Region
export-modulemember -function Read-S3ObjectWithWait
