# Contains logging utilities


# Writes a log to STDOUT and to the log file
# Supports DEBUG, INFO, WARN and ERROR
# @param [String] message The message to log
# @param [String] level The level of log, default to INFO
# @param [Bool] openPDIncident: Whether to open a pagerduty incident if the log is >= ERROR
Function Write-Log() {
    param(
        [string]$message,
        [string]$level = "INFO",
        [string]$application = "unknown",
        [bool]$openPDIncident = $true
    )

    $logDirectory = "C:\Users\selenium\AppData\Roaming\Grid"
    $logFile = "provisioner.log"
    $logPath = "$logDirectory\$logFile"

    # Create the directory to hold logs if it doesn't exist
    if(!(Test-Path -Path $logDirectory )){
        New-Item -ItemType "directory" -Path $logDirectory
    }

    # Create a Powershell Object for the log
    $logObject = New-Object -TypeName psobject
    $logObject | Add-Member -MemberType NoteProperty -Name time        -Value $(Get-Date -format s)
    $logObject | Add-Member -MemberType NoteProperty -Name level       -Value $level
    $logObject | Add-Member -MemberType NoteProperty -Name message     -Value $message
    $logObject | Add-Member -MemberType NoteProperty -Name application -Value $application

    # Convert to JSON, Write Out to Stdout and a file
    $logJson = $logObject | ConvertTo-Json -Compress
    Write-Host $logJson
    [IO.File]::AppendAllText($logPath, $logJson + "`n") # encoded with UTF-8 No BOM by default

    # Trigger Pagerduty Alert if there was an error
    if($openPDIncident -And $level.ToLower() -eq "error") {
        trigger-pagerduty($message)
    }
}

# Triggers a Pagerduty Alert with a message
# @param [String] message The message to log
function trigger-pagerduty() {
    param(
        [string]$message
    )

    # Load in the C# Module to send a page
    $config_directory = "C:\ProgramData\grid-config"
    $moduleDirectory = "$($config_directory)\modules"
    $moduleName = "CBGModules.dll"
    Add-Type -Path "$moduleDirectory\$moduleName"

    # Trigger the pagerduty incident
    [CBGModules.PagerDuty]::Trigger($message, "error")
}

export-modulemember -function Write-Log
