package com.cbg.selenium.capability_matcher;

import org.openqa.grid.internal.utils.DefaultCapabilityMatcher;
import java.util.Map;
import java.util.logging.Logger;

// Cross Browser Grid Capability Matcher - for doing custom matching configuration
public class CBGCapabilityMatcher extends DefaultCapabilityMatcher {
	private static final Logger log = Logger.getLogger(CBGCapabilityMatcher.class.getName());
	private static String osPlatform = "osPlatform";
	private static String osVersion = "osVersion";
	
	/**
	 * Adds support for passing "osPlatform" (e.g., Windows) and "osVersion" (e.g., 10)
	 * @param nodeCapability The capabilities of the node
	 * @param requestedCapability The capabilities the user requested
	 * @return If the requested capabilities match
	*/
	@Override
	public boolean matches(Map<String, Object> nodeCapability, Map<String, Object> requestedCapability) {
		// First check the default to make sure the base capabilities match
		boolean match = super.matches(nodeCapability, requestedCapability);
		
		// Check if user is requesting a specific os platform, and that the previous match passed
		if (requestedCapability.containsKey(osPlatform) && match) {
			match = matchesOSPlatform(nodeCapability, requestedCapability); // Update the match with results
		}
		
		// Check if the user is requesting a specific OS Version, and that the previous match passed
		if (requestedCapability.containsKey(osVersion) && match) {
			match = matchesOSVersion(nodeCapability, requestedCapability); // Update the match with results
		}
		
		// Return the final match
		return match;
	}
	
	/**
	 * Checks if the OS Version matches. Version Examples: 10, 8, Sierra
	 * @param nodeCapability The capabilities of the node
	 * @param requestedCapability The capabilities the user requested
	 * @return If the requested OS Version matches the node's
	 */
	private boolean matchesOSVersion(Map<String, Object> nodeCapability, Map<String, Object> requestedCapability) {
		return capToStrMatch(nodeCapability, requestedCapability, osVersion);
	}
	
	/**
	 * Checks if the OS Platform matches. Platform Examples: Windows, macOS, Linux
	 * @param nodeCapability The capabilities of the node
	 * @param requestedCapability The capabilities the user requested
	 * @return If the requested OS Platform matches the node's
	 */
	private boolean matchesOSPlatform(Map<String, Object> nodeCapability, Map<String, Object> requestedCapability) {
			return capToStrMatch(nodeCapability, requestedCapability, osPlatform);
	}
	
	/**
	 * @param nodeCapability The capabilities of the node
	 * @param requestedCapability The capabilities the user requested
	 * @param capToMatch The specific capability to compare if matches
	 * @return Whether the capabilities match between the requested and the target node
	 */
	private boolean capToStrMatch(Map<String, Object> nodeCapability, Map<String, Object> requestedCapability, String capToMatch) {
		Boolean match = false; // Initialize variable to track match
		log.fine("Preparing to check for a capability match for cap: " + capToMatch);
		
		// Double check that both the node and requested caps contain the capability key
		if(nodeCapability.containsKey(capToMatch) == false || requestedCapability.containsKey(capToMatch) == false)
		{
			return false; // One of the nodes didn't contain it. Return false to prevent a null exception.
		}
		
		try {
			// Get the capabilities for requested & node. Convert to string and ignore string casing.
			String reqCap = requestedCapability.get(capToMatch).toString();
			String nodeCap = nodeCapability.get(capToMatch).toString();
			match = reqCap.equalsIgnoreCase(nodeCap);
			
			log.fine("Requested Capability <" + reqCap + "> matches node capability <" + nodeCap + ">: " + Boolean.toString(match));
			return match;
		} catch (NullPointerException e) {
			log.warning("While processing capabilities, encountered " + e.getClass().toString() + e.getMessage());
			return false;
		}
	}
}
