package com.cbg.selenium.logger;

import java.util.logging.*;

import com.cbg.utils.PagerDutyUtils;

/**
 * This handler will forward log messages of WARNING or SEVERE to PagerDuty
 */
public class PagerDutyLogHandler extends Handler {

    private static final Logger log = Logger.getLogger(PagerDutyLogHandler.class.getName());
    private boolean triggerPagerDutyEvents = true; // Determines if events should be sent to PagerDuty

    public PagerDutyLogHandler() {
        configure();
    }

    // Configures the logger, reads various properties if given
    private void configure() {
        LogManager manager = LogManager.getLogManager();
        String cname = getClass().getName();

        // Read the properties

        //.triggerPagerDutyEvents Enables/Disables sending events & alerts to PagerDuty
        String triggerEventsUserInput = manager.getProperty(cname + ".triggerPagerDutyEvents");
        if (triggerEventsUserInput == null || triggerEventsUserInput == "") {
            // If null or blank, default to true
            triggerPagerDutyEvents = true;
            log.info(String.format("User did not specify 'triggerPagerDutyEvents'. " +
                    "Set triggerPagerDutyEvents to: %s", triggerPagerDutyEvents));
        } else { // If provided, evaluate it to a bool
            triggerPagerDutyEvents = triggerEventsUserInput.equals("true"); // Convert to bool
            log.info(String.format("Set triggerPagerDutyEvents to: %s", triggerPagerDutyEvents));
        }
    }

    @Override
    public void publish(final LogRecord record) {
        // Start a thread to send to PagerDuty, so that any timeouts don't cause performance issues
        new Thread(new Runnable() {
            public void run(){
                sendToPagerDuty(record);
            }
        }).start();
    }

    @Override
    public void flush() {
    }

    @Override
    public void close() throws SecurityException {
    }

    /**
     * Sends a Log Record of WARNING/SEVERE to PagerDuty
     * @param record The log record to send
     */
    private void sendToPagerDuty(LogRecord record) {
        Level level = record.getLevel();

        if(level == Level.WARNING || level == Level.SEVERE) {

            // Check if user requested to trigger events
            if (!triggerPagerDutyEvents) { // If false, return without triggering
                log.info("Not triggering PagerDuty Event as user specified not to. Message Contents was: "
                        + record.getMessage());
                return;
            }

            PagerDutyUtils.trigger(level, record.getMessage());
        }
    }
}
