package com.cbg.selenium.objects;

import java.util.Map;
import java.util.logging.Logger;

import com.cbg.selenium.capabilities.DesktopCapabilities;

/**
 * Object that contains all of Cross Browser Grid's Capabilities
 */
public class CBGCapabilities {

    private DesktopCapabilities desktopCapabilities; // Stores the Desktop Specific Capabilities
    private Map<String, Object> cbgCapabilities; // Stores all of the CBG Capabilities

    private final String capabilityIdentifier = "cbg"; // Identifies the name provided in the capabilities
    private static final Logger log = Logger.getLogger(CBGCapabilities.class.getName());

    /**
     * Constructs a CBGCapabilities Object. This will contain any custom CBG Capabilities,
     * including an embeded DesktopCapabilities object if applicable
     *
     * Currently only supports DesktopCapabilities. In the future this will probably contain more
     * @param requestedCapability The full Selenium Requested Capabilities from the Client
     */
    public CBGCapabilities(Map<String, Object> requestedCapability) {
        if (!requestedCapability.containsKey(capabilityIdentifier)) {
            log.fine("Not creating object as the capability identifier " + capabilityIdentifier + " was not provided.");
            return; // Full Capabilities don't contain the cbg caps. Return
        }

        if (!(requestedCapability.get(capabilityIdentifier) instanceof Map)) { // If it is NOT an instanceOf Map
            log.warning("Not creating object as the capabilities were not an instance of a map"); // This is not expected
            return; // Protect against TypeCast issues and return
        }

        cbgCapabilities = (Map<String, Object>) requestedCapability.get(capabilityIdentifier);
        desktopCapabilities = new DesktopCapabilities(cbgCapabilities); // Parse Desktop Specific Capabilities
    }

    /**
     * @return The desktop specific capabilities
     */
    public DesktopCapabilities getDesktopCapabilities() {
        return desktopCapabilities;
    }

    /**
     * @return If the object is completely empty in attributes
     */
    public boolean empty() {
        return (desktopCapabilities == null); // TODO: Dynamically do this?
    }
}
