package com.cbg.selenium.objects;

import java.util.Map;
import java.util.LinkedHashMap;
import java.util.logging.Logger;

/**
 * Object that contains all of Chromedriver's Capabilities
 */
public class ChromeOptCapabilities {

    private LinkedHashMap<String, Object> chromeOptions; // Stores all of the ChromeDriver Options
    private static final Logger log = Logger.getLogger(ChromeOptCapabilities.class.getName());
    private final String capabilityIdentifier = "chromeOptions"; // Identifies the name provided in the capabilities
    private final String binaryKey = "binary"; // Identifies the map key for the ChromeOpts Binary

    /**
     * Constructs a ChromeOptCapabilities Object
     * @param fullCapabilities The full Selenium Requested Capabilities from the Client
     */
    public ChromeOptCapabilities(Map<String, Object> fullCapabilities) {
        chromeOptions = parseChromeCaps(fullCapabilities);
    }

    /**
     * Parses a client's requested Capabilities for the ChromeDriver Specific Options
     * @param caps The full requested capabilities passed by the client
     * @return The ChromeDriver specific options
     */
    private LinkedHashMap<String, Object> parseChromeCaps(Map<String, Object> caps) {
        LinkedHashMap<String, Object> chromeOpts = new LinkedHashMap<String, Object>();

        // If Caps does NOT already contain chromeOptions, return the empty object. Nothing to parse.
        if(!caps.containsKey(capabilityIdentifier)) { return chromeOpts; }

        // Grab the ChromeOptions from within the full Selenium capabilities
        try {
            chromeOpts = (LinkedHashMap<String, Object>) caps.get(capabilityIdentifier);
        } catch (ClassCastException e) {
            log.warning("Could not cast ChromeOptions. Exception: " + e.toString());
        }

        return chromeOpts;
    }

    /**
     * @return Chrome specific options
     */
    public Map<String, Object> getChromeOptions() {
        return chromeOptions;
    }

    /**
     * Returns the full Selenium Capabilities with the Chrome Options created from this object
     * @param origCaps The original Selenium Capabilities from the requester
     * @return A map of all of the capabilities to send to the node
     */
    public Map<String, Object> getFullCapabilities(Map<String, Object> origCaps) {
        if(!chromeOptions.isEmpty()) { // If there are ChromeDriver Options to append (NOT empty)
            origCaps.put(capabilityIdentifier, chromeOptions);
        }
        return origCaps;
    }

    /**
     * @param path The Binary Path for Chrome to launch with
     */
    public void setBinaryPath(String path) {
        chromeOptions.put(binaryKey, path);
    }

    /**
     * @return The Binary Path for Chrome to launch with
     */
    public String getBinaryPath() {
        if (chromeOptions.containsKey(binaryKey)) {
            return (String) chromeOptions.get(binaryKey);
        } else {
            return null; // Being a bit extra safe, even though the cast should handle this.
        }
    }
}
