package com.cbg.selenium.objects;

import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.logging.Logger;
import java.util.zip.ZipException;

import org.apache.commons.io.FileUtils;

import com.cbg.utils.DesktopFileUtils;
import com.cbg.exceptions.UnauthorizedURLException;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.lang3.ArrayUtils;

/**
 * Object for the StandaloneApp which will be used to test against
 * @author Dylan Reichstadt
 */
public class StandaloneApp {

	/*
	Store the parent folder that the app will live
	Originally planned to store this as a user-defined config. However, Hub and Node both need this. What if Hub and Node had a different value?
	Reverted (sad day), see detailed comments here:
	https://jira.twitch.com/browse/QE-2197?focusedCommentId=501031&page=com.atlassian.jira.plugin.system.issuetabpanels%3Acomment-tabpanel#comment-501031
	 */
	private static String rootFolder = "C:\\Program Files\\Twitch";
	private static final Logger log = Logger.getLogger(StandaloneApp.class.getName());
	private String[] whitelistedDownloadHosts = { "desktop-builds.clients.internal.justin.tv" };
	private static final String unknownFolderIdentifier = "unknown";

	/**
	 * @return The root folder that holds the application. Null if empty / doesn't exist
	 */
	public static String getRootFolder() {
		if (rootFolder == null) {
			return null;
		} else if (rootFolder.isEmpty()) {
			return null;
		} else {
			return rootFolder;
		}
	}

	/**
	 * Determines the folder to store the artifact files into
	 * @param downloadURL The URL to download the file from
	 * @return The folder to extract artifact contents into, null if there was an error
	 */
	public static String getArtifactExtractionFolder(String downloadURL) {
		String folder = null;
		try {
			String filename = new File(new URL(downloadURL).getPath()).getName(); // Get the full filename
			folder = FilenameUtils.removeExtension(filename); // Strip the extension to store in the folder
		} catch (Exception e) { }
		return folder;
	}

	/**
	 * @param downloadURL The URL to download the file from
	 * @return The full path to where the artifacts are stored, built off a URL
	 */
	public static String buildFullArtifactPath(String downloadURL) {
		String extractFolder = getArtifactExtractionFolder(downloadURL);

		// ERROR HANDLING! This block shouldn't be called in an ideal situation.
		if (StandaloneApp.getRootFolder() == null || StandaloneApp.getRootFolder().isEmpty() ||
				extractFolder == null || extractFolder.isEmpty() ) {
			// If folder is null/empty, return unknown
			log.warning("Could not identify root/extract folder, so returning " + unknownFolderIdentifier);
			return unknownFolderIdentifier;
		} else {
			return StandaloneApp.getRootFolder() + "\\" + extractFolder;
		}
	}


	URL downloadURL; // Store where to download the file from
	String artifactPath; // Where artifacts are stored
	File downloadFile; // Store the downloaded file

	// TODO: Expose the file object for extracted file

	/**
	 * Builds a StandaloneApp object which is intended to be the application used for testing
	 * @param downloadURL The URL to download the application from
	 */
	public StandaloneApp(String downloadURL) throws MalformedURLException {
		this.downloadURL = new URL(downloadURL);
		this.artifactPath = buildFullArtifactPath(downloadURL);
	}

	/**
	 * Downloads an application from the server and stores it within the root folder
	 * @throws MalformedURLException
	 * @throws IOException
	 */
	public void download() throws IOException, UnauthorizedURLException {
		if(!isAuthorizedDownloadURL()) { throw new UnauthorizedURLException(); } // Authenticate with Whitelist
		if(getRootFolder() == null) {
			log.severe("Unable to download StandaloneApp. No root folder was provided.");
			throw new IOException();
		}

		long startTime = System.currentTimeMillis(); // Measure Duration

		// Download the app
		this.downloadFile = DesktopFileUtils.downloadFromURL(this.downloadURL, getRootFolder());
	}

	/**
	 * Extracts the application and cleans up its download files
	 * @throws ZipException
	 * @throws IOException
	 */
	public void install() throws ZipException, IOException {
		long startTime = System.currentTimeMillis(); // Measure Duration

		if (this.downloadFile == null || !this.downloadFile.exists()) {
			log.severe("Unable to install StandaloneApp. Problem with the downloaded file.");
			throw new IOException();
		}

		// Extract the app
		DesktopFileUtils.extractBinary(this.downloadFile, getArtifactPath());

		// Clean up the zip file that was downloaded
		log.info(String.format("Deleting %s", this.downloadFile));
		FileUtils.deleteQuietly(this.downloadFile);
		this.downloadFile = null;
	}

	/**
	 * Removes all application files
	 * @throws IOException
	 */
	public void destroy() throws IOException {
		long startTime = System.currentTimeMillis(); // Measure Duration

		if (getRootFolder() == null) {
			log.warning("Unable to delete StandaloneApp as root folder is not provided.");
			return;
		}

		// Delete the app
		DesktopFileUtils.deleteDirectoryWithRetry(getArtifactPath());
	}

	public URL getDownloadURL() {
		return this.downloadURL;
	}

	/**
	 * @return The path where the artifacts are stored
	 */
	public String getArtifactPath() {
		return this.artifactPath;
	}

	/**
	 * @return If the artifacts already exist
	 */
	public Boolean exists() {
		if (getArtifactPath().equals(unknownFolderIdentifier)) {
			log.warning("Going to overwrite the unknown artifact folder");
			return false; // Overwrite any unknown folders to be safe
		}

		return new File(getArtifactPath()).exists();
	}

	/**
	 * Ensures a Download URL's Host is within the Whitelist
	 * @return Whether the download url's host is within the whitelist
	 */
	public Boolean isAuthorizedDownloadURL() {
		return ArrayUtils.contains(whitelistedDownloadHosts, this.downloadURL.getHost());
	}
}
