package com.cbg.selenium.proxy;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.logging.Logger;

import com.cbg.selenium.objects.StandaloneApp;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.apache.http.Consts;
import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.NameValuePair;
import org.apache.http.client.ClientProtocolException;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.HttpDelete;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.util.EntityUtils;
import org.openqa.grid.common.RegistrationRequest;
import org.openqa.grid.internal.GridRegistry;
import org.openqa.grid.internal.TestSession;
import org.openqa.grid.selenium.proxy.DefaultRemoteProxy;

import com.cbg.selenium.servlets.DesktopApp;
import com.cbg.selenium.objects.CBGCapabilities;
import com.cbg.selenium.objects.ChromeOptCapabilities;
import com.cbg.utils.ChromeOptUtils;

import java.util.Map;

/**
 * This class extends the DefaultRemoteProxy of Selenium.
 * It adds support for Desktop Application testing, by installing the application on the node prior to the test session
 *   and deleting the application from the node after the test session.
 * @author Dylan Reichstadt
 *
 */
public class CBGRemoteProxy extends DefaultRemoteProxy {
	
	private static final int CONNECTION_TIMEOUT = (int) TimeUnit.SECONDS.toMillis(120); // TODO: Pull from Selenium Configuration perhaps for timeouts
	private static final Logger log = Logger.getLogger(CBGRemoteProxy.class.getName());

	public CBGRemoteProxy(RegistrationRequest arg0, GridRegistry arg1) {
		super(arg0, arg1);
		log.fine("CBG remote proxy has been loaded!");
	}

	@Override
	/**
	 * This will run before the hub gets a session
	 */
	public TestSession getNewSession(Map<String, Object> requestedCapability) {
		// Parse CBG Related Capabilities
		CBGCapabilities cbgCaps = new CBGCapabilities(requestedCapability);

		// if there are desktop capabilities, merge them into ChromeDriver Caps
		if (requestIsADesktopAppTest(cbgCaps)) {
			requestedCapability = mergeNewChromeDriverCapabilities(requestedCapability, cbgCaps);
		}

		return super.getNewSession(requestedCapability);
	}

	/**
	 * This method runs on the hub before a session is started
	 * @param session The test session created from the request
	 */
	@Override
	public void beforeSession(TestSession session) {
		super.beforeSession(session);
		CBGCapabilities cbgCaps = new CBGCapabilities(session.getRequestedCapabilities());

		// If the app binary url is present in the capabilities, request to install it on the node
		if (requestIsADesktopAppTest(cbgCaps)) {
			requestApplicationInstallation(cbgCaps.getDesktopCapabilities().getartifactURL());
		}
	  }

	@Override
	public void afterSession(TestSession session) {
		CBGCapabilities cbgCaps = new CBGCapabilities(session.getRequestedCapabilities());

		// If a desktop session, delete it
		if (requestIsADesktopAppTest(cbgCaps)) {
			requestApplicationDeletion(true, false); // Do not uninstall app, allow app to be cached
		}
		super.afterSession(session);
	}
	
	/**
	 * Makes a request for the node to install an application for testing
	 * @param artifactURL The url to download the application
	 */
	private void requestApplicationInstallation(String artifactURL) {
	    // Create the client
	    CloseableHttpClient client = getHTTPClient();
	    
	    // Create the POST request to install the application
	    String url = buildDesktopAppEndpoint();
	    HttpPost post = new HttpPost(url);
	    
	    // Create the POST Data
	    UrlEncodedFormEntity entity = createApplicationInstallPostData(artifactURL);
	    post.setEntity(entity); // Add the content to the POST Request
	    
	    // Make the request.
	    process(client, post);
	}
	
	/**
	 * Makes a request for the node to delete the application used for testing
	 * @param killProcess If the node should hard-kill the process
	 * @param uninstall If the node should uninstall and delete the artifact
	 */
	private void requestApplicationDeletion(Boolean killProcess, Boolean uninstall) {
	    // Create the client
	    CloseableHttpClient client = getHTTPClient();
	    
	    // Create the POST request to delete the application
	    String url = buildDesktopAppEndpoint();
	    url += String.format("?kill=%b&uninstall=%b", killProcess, uninstall);
	    HttpDelete deleteRequest = new HttpDelete(url);
	    
	    // Make the request
	    process(client, deleteRequest);
	}
	
	/**
	 * Processes an HTTP Request
	 * @param client The HTTP Client
	 * @param request The HTTP Request to make
	 */
	private void process(CloseableHttpClient client, HttpRequestBase request) {
		// Make the request
		try {
			log.info(String.format("-> %s %s", request.getMethod(), request.getURI()));
			HttpResponse response = client.execute(request);
			log.info(String.format("<- [%s] | %s", response, request.getURI()));

			// Make sure the response was good. If it wasn't, throw an error to stop the session from starting.
			if (response.getStatusLine().getStatusCode() != 204) {
				HttpEntity entity = response.getEntity();
				throw new java.lang.RuntimeException(String.format("Unexpected response code %d. Path: %s | Message: %s", response.getStatusLine().getStatusCode(), request.getURI(), EntityUtils.toString(entity)));
			}
		} catch (ClientProtocolException e) {
			log.warning(String.format("Problem processing request %s %s. Received Back: %s", request.getMethod(), request.getURI().toString(), ExceptionUtils.getStackTrace(e)));
		} catch (IOException e) {
			log.warning(String.format("Problem processing request %s %s. Received Back: %s", request.getMethod(), request.getURI().toString(), ExceptionUtils.getStackTrace(e)));
		} finally {
			try { client.close(); } catch (Exception e) {
				log.warning(String.format("Problem closing request client. Received Back: %s", ExceptionUtils.getStackTrace(e)));
			}
		}
	}
	
	/**
	 * Creates POST parameters for the application install request
	 * @param artifactURL The application url to download
	 * @return A list of parameters to be used in the POST request for application installation
	 */
	private UrlEncodedFormEntity createApplicationInstallPostData(String artifactURL) {
	    List<NameValuePair> formparams = new ArrayList<NameValuePair>(); // Initialize an array list
	    formparams.add(new BasicNameValuePair("artifactURL", artifactURL)); // Add it to the array
	    return new UrlEncodedFormEntity(formparams, Consts.UTF_8);
	}
	
	/**
	 * Builds the endpoint for installing/uninstalling the deskotp application on the node
	 * @return The endpoint used to install and uninstall applications
	 */
	private String buildDesktopAppEndpoint() {
		return String.format("http://%s:%d/extra/%s", getRemoteHost().getHost(), getRemoteHost().getPort(), DesktopApp.class.getSimpleName());
	}
	
	/**
	 * Builds an HTTP Client that can be used for requests
	 * @return An HTTP Client with connection timeouts configured
	 */
	private CloseableHttpClient getHTTPClient() {
		RequestConfig requestConfig = RequestConfig.custom().setConnectTimeout(CONNECTION_TIMEOUT)
                .setSocketTimeout(CONNECTION_TIMEOUT).build();
		return HttpClientBuilder.create().setDefaultRequestConfig(requestConfig).build();
	}

	/**
	 * Merges any custom Cross Browser Grid capabilities into ChromeDriver Options
	 * An example is when we need to provide ChromeDriver with a desktop application's path.
	 * @param requestedCaps The capabilities given to us from the client
	 * @param cbgCaps The Cross Browser Specific capabilities created upstream
	 * @return The full capabilities to pass back to a node, with ChromeDriver options modified if needed.
	 */
	private Map<String, Object> mergeNewChromeDriverCapabilities(Map<String, Object> requestedCaps, CBGCapabilities cbgCaps) {
		ChromeOptCapabilities cOpts = new ChromeOptCapabilities(requestedCaps); // Build ChromeDriver Object from Capabilities

		// Merge desktop capabilities into options
		ChromeOptUtils.mergeDesktopBinaryFromCapabilities(cbgCaps,
				cOpts,
				StandaloneApp.buildFullArtifactPath(cbgCaps.getDesktopCapabilities().getartifactURL()));

		log.info(String.format("Merged Desktop Capabilities into Chrome Options. New Chrome Options: %s", cOpts.getChromeOptions()));
		return cOpts.getFullCapabilities(requestedCaps); // Return the full Capabilities & Options to give to the node
	}

	/**
	 * Determines if the session request is a Desktop Application Test Request
	 * @param cbgCaps The Cross Browser Grid specific Capabilities
	 * @return Whether the request is a desktop application test request
	 */
	public static boolean requestIsADesktopAppTest(CBGCapabilities cbgCaps) {
        try {
            return (!cbgCaps.getDesktopCapabilities().empty() && // There are Desktop Capabilities (not empty)
					cbgCaps.getDesktopCapabilities().getartifactURL() != null && // App Binary URL is not null
					!cbgCaps.getDesktopCapabilities().getartifactURL().isEmpty()); // App Binary URL is not empty
        } catch (NullPointerException e) { // Raised if cbgCaps is empty
            return false;
        }
	}
}
