package com.cbg.selenium.servlets;

import java.io.IOException;
import java.io.PrintWriter;
import java.util.logging.Logger;
import java.util.zip.ZipException;

import javax.servlet.ServletException;

import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import com.cbg.utils.DesktopFileUtils;
import org.apache.commons.lang3.SystemUtils;
import org.apache.commons.lang3.exception.ExceptionUtils;

import com.cbg.selenium.objects.StandaloneApp;
import com.cbg.exceptions.UnauthorizedURLException;

/**
 * This is an HTTP Servlet that adds support for the installation and uninstallation of a desktop application
 * By allowing this, desktop applications can be tested.
 * The path to this servlet on the node is /extra/DesktopApp
 * @author Dylan Reichstadt
 *
 */
public class DesktopApp extends HttpServlet {
	private static final long serialVersionUID = 8625877641262288475L;
	private static final Logger log = Logger.getLogger(DesktopApp.class.getName());
	
	private StandaloneApp app; // Store the application in the node's state

	/**
	 * Installs a desktop application from the body parameter "artifactURL"
	 */
	public void doPost(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {
		// Check to make sure artifactURL was provided in the body of the request
		if (!downloadURLIsPresent(request)) {
			response = returnErrorFromEnum(response, Error.MissingArtifactURL);
			return;
		}

		// Create the application object
		String downloadURL = getDownloadURL(request);
		try {
			this.app = new StandaloneApp(downloadURL);
		} catch (Exception e) { // Expected. Likely user provided a bad URL.
			response = returnErrorFromEnum(response, Error.MalformedURL);
			log.info(String.format("When downloading from URL [%s], encountered exception: %s", downloadURL,
					ExceptionUtils.getStackTrace(e)));
			return;
		}

		if(!app.exists()) {
			// Download the file from the url requested
			try {
				app.download();
			} catch (IOException e) { // Not expected case
				response = returnErrorFromEnum(response, Error.IOException);
				log.warning(String.format("When downloading from URL [%s], encountered exception: %s", app.getDownloadURL(), ExceptionUtils.getStackTrace(e)));
				return;
			} catch (UnauthorizedURLException e) {
				response = returnErrorFromEnum(response, Error.UnauthorizedURL);
				log.info(String.format("Unauthorized URL [%s] was attempted to be downloaded", app.getDownloadURL()));
				return;
			}

			// Install the application
			try {
				app.install();
			} catch (ZipException e) { // Expected. Likely the user provided a download that was not zip.
				response = returnErrorFromEnum(response, Error.ZipException);
				log.info(String.format("When extracting the file, encountered exception: %s", ExceptionUtils.getStackTrace(e)));

				// Clean up! There was a problem extracting so make sure everything is deleted.
				try {
					destroyApp();
				} catch (IOException eCleanup) {
					log.warning(String.format("There was a problem cleaning up and deleting the app files. Exception: %s", ExceptionUtils.getStackTrace(eCleanup)));
				}
				return;
			} catch (IOException e) {
				response = returnErrorFromEnum(response, Error.IOException);
				log.warning(String.format("When extracting the file, encountered exception: %s", ExceptionUtils.getStackTrace(e)));

				// Clean up! There was a problem extracting so make sure everything is deleted.
				try {
					destroyApp();
				} catch (IOException eCleanup) {
					log.warning(String.format("There was a problem cleaning up and deleting the app files. Exception: %s", ExceptionUtils.getStackTrace(eCleanup)));
				}
				return;
			}
		} else {
			log.info("Application already exists. Skipping Download/Install");
		}
		
		// Return
		log.info("Finished preparing the application. Returning.");
		response.setStatus(204);
	}
	
	/**
	 * Uninstalls/Removes a desktop application from the machine
	 * Endpoint accepts two params: uninstall, kill
	 * Uninstall will remove the app from the machine
	 * Kill will terminate the known processes forcefully
	 */
	public void doDelete(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {
		final String[] processNames = new String[]{ "TwitchUI.exe", "TwitchAgent.exe" }; // Known processes

		// Get the parameters
		String uninstall = request.getParameter("uninstall");
		String kill = request.getParameter("kill");
		log.fine(String.format("Params: Uninstall %s Kill %s", uninstall, kill));

		// If user specifies to kill processes, kill known processes
		// Default to false
		if (kill != null && kill.equals("true")) {
			log.info("Killing Desktop processes");
			String command;
			for (int i=0; i < processNames.length; i++) {

				// Build command based on OS
				if (SystemUtils.IS_OS_WINDOWS) {
					command = String.format("taskkill /F /IM %s", processNames[i]);
				} else if (SystemUtils.IS_OS_MAC || SystemUtils.IS_OS_LINUX) {
					command = String.format("pkill \"%s\"", processNames[i]);
				} else {
					log.severe("Could not determine OS for killing process. OS Name: " + SystemUtils.OS_NAME);
					returnErrorFromEnum(response, Error.CannotKillProcessUnknownOS);
					return;
				}

				log.fine("Executing " + command);
				Runtime.getRuntime().exec(command);
			}

			// Just for safety....
			try {
				Thread.sleep(300);
			} catch (Exception e) {}
		}

		// If user specified to uninstall the app, uninstall it
		// Default to true if not provided
		if (uninstall == null || uninstall.equals("true")) {
			log.info("Uninstalling Desktop Application");

			try {
				DesktopFileUtils.deleteDirectoryWithRetry(StandaloneApp.getRootFolder());
			} catch (IOException e) {
				log.warning(String.format("There was a problem cleaning up and deleting the app files. Exception: %s",
						ExceptionUtils.getStackTrace(e)));
				response = returnErrorFromEnum(response, Error.IOException);
				return;
			}
		}

		response.setStatus(204);
	}
	
	// Destroys an application and sets its reference to null
	private void destroyApp() throws IOException {
	    if(app != null) { // If the application exists, destroy it
	        this.app.destroy();
	        this.app = null;
	    }
	}
	
	/**
	 * Checks if the download url was provided within the request parameters
	 * @param request The request
	 * @return True if the download url was present
	 */
	private boolean downloadURLIsPresent(HttpServletRequest request) {
		String artifactURL = getDownloadURL(request);
		return (artifactURL != null && !(artifactURL.isEmpty())); // URL is not null, and is not empty
	}
	
	/**
	 * Gets the download url provided within the request
	 * @param request The request
	 * @return The url to the download
	 */
	private String getDownloadURL(HttpServletRequest request) {
		return request.getParameter("artifactURL");
	}

	/**
	 * Helper method that will set the status and error message from a DesktopAppError Enum
	 * @param response The HTTP Response Object to send back to the user
	 * @param err The error class
	 * @return An object to return to the user with the response code and error message provided
	 * @throws IOException
	 */
	private HttpServletResponse returnErrorFromEnum(HttpServletResponse response, Error err) throws IOException {
		return returnError(response, err.getStatusCode(), err.getDescription());
	}
	
	/**
	 * Helper method that will set the status and return an error message in json format
	 * @param response The HTTP Response Object to send back to the user
	 * @param statusCode The status/response code to return
	 * @param errorMsg The error message to provide to the user
	 * @return An object to return to the user with the response code and error message provided
	 * @throws IOException
	 */
	private HttpServletResponse returnError(HttpServletResponse response, int statusCode, String errorMsg) throws IOException {
		response.setStatus(statusCode);
		response.setContentType("application/json");
		PrintWriter out = response.getWriter();
		out.print(String.format("{\"error\":\"%s\"}", errorMsg));
		return response;
	}

	/**
	 * Stores Error Status Codes and Descriptions to return to the user
	 */
	public enum Error {
		MalformedURL(400, "The provided artifact url was not valid"),
		IOException(500, "There was a problem processing the artifact file. Please contact the server administrator"),
		CannotKillProcessUnknownOS(500, "There was a problem killing the desktop application process. Could not determine the OS"),
		ZipException(400, "There was a problem extracting the download contents. Ensure contents are of type ZIP"),
		UnauthorizedURL(422, "The artifactURL is not valid. Contact the server administrator."),
		MissingArtifactURL(422, "The required parameter artifactURL was not provided");

		private final int statusCode;
		private final String description;

		Error(int statusCode, String description) {
			this.statusCode = statusCode;
			this.description = description;
		}

		public String getDescription() {
			return description;
		}

		public int getStatusCode() {
			return statusCode;
		}
	}
}
