package com.cbg.utils;

import com.cbg.configs.CBGConfig;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.dataformat.yaml.YAMLFactory;

import java.io.File;
import java.util.logging.Logger;

/**
 * Utilities for handling Cross Browser Grid Configuration
 * To load a configuration file, provide java -Dcbg.config.path=path_to_cbg_config.yaml
 * Configuration must be in yaml format
 */
public class ConfigUtils {
    private static CBGConfig config;
    private static String configSystemProperty = "cbg.config.path";
    private static final Logger log = Logger.getLogger(ConfigUtils.class.getName());

    /**
     * @return A configuration object
     */
    public static CBGConfig getConfig() {
        if(config == null) { // If it's null, initialize it.
            config = readConfig();
        }

        return config;
    }

    /**
     * Reads configuration from a file
     * @return The configuration from the file
     */
    private static CBGConfig readConfig() {
        ObjectMapper mapper = new ObjectMapper(new YAMLFactory());
        CBGConfig configuration = new CBGConfig();

        File configFile = getConfigFile();
        if (configFile == null) {
            log.info("A configuration file was not provided. Skipping import");
            return configuration;
        }

        try {
            configuration = mapper.readValue(configFile, CBGConfig.class);
            log.info(String.format("Successfully loaded CBG Config from %s", configFile.toString()));
        } catch (Exception e) {
            log.warning("While reading config, encountered exception: " + e.getMessage());
        }

        return configuration;
    }

    /**
     * Reads a configuration file given from a system environment variable, with the key "cbg.config.path"
     * @return A configuration file provided from the configuration with key "cbg.config.path"
     */
    private static File getConfigFile() {
        String configPath = System.getProperty(configSystemProperty);

        if (configPath == null || configPath.isEmpty()) { return null; }
        return new File(configPath);
    }
}
