package com.cbg.utils;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Enumeration;
import java.util.concurrent.TimeUnit;
import java.util.logging.Logger;
import java.util.zip.ZipEntry;
import java.util.zip.ZipException;
import java.util.zip.ZipFile;

import org.apache.commons.io.FileUtils;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.exception.ExceptionUtils;

/**
 * Contains utilities for downloading, extracting, and removing app program files
 * @author Dylan Reichstadt
 *
 */
public class DesktopFileUtils {
	
	private static final Logger log = Logger.getLogger(DesktopFileUtils.class.getName());
	
	/**
	 * Downloads a file from the url to a destination
	 * @param downloadURL The url to download
	 * @param destinationDirectory The directory to download the file to
	 * @return The File that was downloaded
	 * @throws MalformedURLException Raised when a URL was invalid
	 * @throws IOException Raised when there was a problem downloading that application
	 */
	public static File downloadFromURL(URL downloadURL, String destinationDirectory) throws MalformedURLException, IOException {
		String filename = FilenameUtils.getName(downloadURL.getPath()); // Grab the name of the file being downloaded
		String destinationPath = String.format("%s\\%s", destinationDirectory, filename); // Create the full path to that file
		
		File destination = new File(destinationPath);
		destination.getParentFile().mkdirs(); // Make the directory path
		
		log.info(String.format("Downloading URL %s to Destination %s", downloadURL, destination));
		FileUtils.copyURLToFile(downloadURL, destination);
		
		return destination;
	}
	
	/**
	 * Extracts a ZIP file contents to the directory, and then deletes the source file
	 * @param sourceFile The source zip file
	 * @throws ZipException Occurs when there was a problem unzipping the application
	 * @throws IOException Occurs when there was a problem writing to the disk
	 */
	public static void extractBinary(File sourceFile, String destinationPath) throws ZipException, IOException {
		ZipFile zipFile = new ZipFile(sourceFile); // Create reference to the zip file. It'll throw an exception if it isn't.

		File extractDestination = new File(destinationPath);
		extractDestination.mkdirs();

		log.info(String.format("Unzipping the file %s to %s", sourceFile, extractDestination));
		
		// Extract the contents to the parent directory of the source
		try {
			Enumeration<? extends ZipEntry> entries = zipFile.entries();
			while (entries.hasMoreElements()) {
				ZipEntry entry = entries.nextElement();
				File entryDestination = new File(extractDestination, entry.getName());
				if (entry.isDirectory()) {
					entryDestination.mkdirs();
				} else {
					entryDestination.getParentFile().mkdirs();
					InputStream in = zipFile.getInputStream(entry);
					OutputStream out = new FileOutputStream(entryDestination);
					IOUtils.copy(in, out);
					in.close();
					out.close();
				}
			}
		} finally {
			zipFile.close();
		}
		
		log.info(String.format("%s successfully unzipped to %s.", sourceFile, extractDestination));
	}
	
	/**
	 * Deletes a directory and retries up to 10 times
	 * @param directory The directory to delete
	 * @throws IOException When there was a problem deleting the directory
	 */
	public static void deleteDirectoryWithRetry(String directory) throws IOException {
	    File appDir = new File(directory);
	    log.info(String.format("Deleting Directory: %s", appDir));
		
	    // Set up max attempts
	    int attempts = 0;
	    int maxAttempts = 10;
	    
	    while(attempts < maxAttempts) {
	    		try {
	    			attempts += 1;
	    			FileUtils.deleteDirectory(appDir);
	    			log.info(String.format("Directory deleted: %s", appDir));
	    			break; // It was successful, get out of the loop
	    		} catch (IOException e) {
	    			log.fine(String.format("[%d/%d] Encountered exception when deleting %s. Exception: %s", attempts, maxAttempts, appDir, ExceptionUtils.getStackTrace(e)));
	    			if(attempts >= maxAttempts) {
	    				throw e;
	    			} else {
	    				// Sleep for a second and then retry
	    				try {
						TimeUnit.SECONDS.sleep(1);
					} catch (InterruptedException e1) {
						log.warning("While sleeping, encountered exception "+ ExceptionUtils.getStackTrace(e1));
					}
	    			}
	    		}
	    }
	}

}
