package com.cbg.utils;

import java.time.OffsetDateTime;
import java.util.logging.Level;
import java.util.logging.Logger;

import com.cbg.configs.CBGConfig;
import com.github.dikhan.pagerduty.client.events.*;
import com.github.dikhan.pagerduty.client.events.exceptions.*;
import com.github.dikhan.pagerduty.client.events.domain.*;
import org.json.JSONObject;

public class PagerDutyUtils {

    private static final PagerDutyEventsClient client = PagerDutyEventsClient.create();
    private static CBGConfig config = ConfigUtils.getConfig();
    private static final Logger log = Logger.getLogger(PagerDutyUtils.class.getName());

    public static PagerDutyEventsClient getClient() {
        return client;
    }

    /**
     * Triggers a PagerDuty Incident
     * @param level The log level to trigger as
     * @param message The message to include in the incident
     */
    public static void trigger(Level level, String message) {
        String instanceID = config.getInstanceID();
        String serviceKey = config.getPagerdutyServiceKey();

        if(serviceKey == null) {
            log.info("Routing service key was null. Unable to route PagerDuty Incident"); // Not using log warn, infinite loop...
            return;
        }

        Payload payload = Payload.Builder.newBuilder()
                .setSummary(getSummary(message))
                .setSource(instanceID)
                .setSeverity(convertLevelToSeverity(level))
                .setTimestamp(OffsetDateTime.now())
                .setCustomDetails(getCustomDetails(message))
                .build();

        TriggerIncident incident = TriggerIncident.TriggerIncidentBuilder
                .newBuilder(serviceKey, payload)
                .build();

        try {
            EventResult event = getClient().trigger(incident);
            log.info("Triggered PagerDuty event: " + event.toString());
        } catch (NotifyEventException e) {
            log.info("Failed to raise PagerDuty notification. " + e.toString()); // Not using logger, infinite loop...
        }
    }

    /**
     * Converts a Log Level to a PagerDuty Severity Level
     * @param level A log level
     * @return A severity representation of that log level
     */
    public static Severity convertLevelToSeverity(Level level) {
        if (level == Level.WARNING) {
            return Severity.WARNING;
        } else if (level == Level.SEVERE) {
            return Severity.CRITICAL;
        } else if (level == Level.INFO) {
            return Severity.INFO;
        } else if (level == Level.FINE || level == Level.FINER || level == Level.FINEST) {
            return Severity.INFO;
        } else {
            return Severity.INFO;
        }
    }

    /**
     * @param message The message to be logged
     * @return A JSON Object containing the full message
     */
    private static JSONObject getCustomDetails(String message) {
        JSONObject details = new JSONObject();
        details.append("Details", message);
        return details;
    }

    /**
     * Returns a shortened version of the error message, as PagerDuty Incidents are max'd at 1024 characters
     * @param message The message to be logged
     * @return A shortened string of the message
     */
    private static String getSummary(String message) {
        if(message.length() >= 101) { // Index starts at 0, so +1 for length
            message = message.substring(0, 100);
        }

        return "CBG Binary Encountered Error. See Details for full exception. Partial Message: " + message;
    }
}
