package com.cbg.selenium.objects;

import static org.junit.Assert.*;

import java.io.IOException;
import java.net.MalformedURLException;

import com.cbg.exceptions.UnauthorizedURLException;
import org.junit.Test;

public class StandaloneAppTest {
    private String malformedURL = "test";
    private String rootUrl = "https://desktop-builds.clients.internal.justin.tv/desklight";
    private String unauthorizedURL = "https://unauthorized-desktop-builds.clients.internal.justin.tv/desklight/pull-27-119-a07177e.dev.windows.zip";
    private String authorizedURL = rootUrl + "/pull-27-119-a07177e.dev.windows.zip";
    private String nonZipURL = rootUrl     + "/pull-27-119-a07177e.dev.windows.rar";
    private String targzURL = rootUrl      + "/pull-27-119-a07177e.dev.windows.tar.gz";

    @Test
    public void withAMalformedURL() {

        try {
            StandaloneApp app = new StandaloneApp(malformedURL);
            fail("Expected a MalformedURLException");
        } catch(MalformedURLException e) {
            // Expected, do nothing
        } catch (Exception e) {
            fail("Expected a MalformedURLException. Got " + e);
        }
    }

    @Test
    public void withANonWhitelistedURL() {
        StandaloneApp app;
        try {
            app = new StandaloneApp(unauthorizedURL);
        } catch(Exception e) {
            fail("Encountered exception: " + e.getMessage());
            return;
        }

        // URL should be invalid
        assertFalse(app.isAuthorizedDownloadURL());

        try {
            app.download();
            fail("Expected UnauthorizedURLException");
        } catch (UnauthorizedURLException e) { }
        catch (IOException e) {
            fail("Encountered exception " + e.getMessage());
        }
    }

    @Test
    public void withAWhitelistedURL() {
        StandaloneApp app;
        try {
            app = new StandaloneApp(authorizedURL);
        } catch(Exception e) {
            fail("Encountered exception: " + e.getMessage());
            return;
        }

        // URL should be valid
        assertTrue(app.isAuthorizedDownloadURL());
    }

    @Test
    public void withANonZipURL() {
        StandaloneApp app;
        try {
            app = new StandaloneApp(nonZipURL);
        } catch (Exception e) {
            fail("Expected no exception. Got " + e);
        }
    }

    @Test
    public void getArtifactExtractionFolderWorks() {
        // A Zip url
        String result = StandaloneApp.getArtifactExtractionFolder(authorizedURL);
        assertEquals("pull-27-119-a07177e.dev.windows", result);

        // A non-zip url
        result = StandaloneApp.getArtifactExtractionFolder(nonZipURL);
        assertEquals("pull-27-119-a07177e.dev.windows", result);

        // a tar.gz url
        result = StandaloneApp.getArtifactExtractionFolder(targzURL);
        assertEquals("pull-27-119-a07177e.dev.windows.tar", result);

        // An invalid url
        result = StandaloneApp.getArtifactExtractionFolder("notAURL");
        assertNull(result);
    }

    @Test
    public void buildFullArtifactPathWorks() {
        // A Zip url
        String result = StandaloneApp.buildFullArtifactPath(authorizedURL);
        assertEquals(StandaloneApp.getRootFolder() + "\\" + "pull-27-119-a07177e.dev.windows", result);

        // A non-zip url
        result = StandaloneApp.buildFullArtifactPath(nonZipURL);
        assertEquals(StandaloneApp.getRootFolder() + "\\" + "pull-27-119-a07177e.dev.windows", result);

        // a tar.gz url
        result = StandaloneApp.buildFullArtifactPath(targzURL);
        assertEquals(StandaloneApp.getRootFolder() + "\\" + "pull-27-119-a07177e.dev.windows.tar", result);

        // An invalid url
        result = StandaloneApp.buildFullArtifactPath("notAURL");
        assertEquals("unknown", result);
    }

    @Test
    public void existWorks() {
        StandaloneApp app;
        try {
            app = new StandaloneApp(authorizedURL);
        } catch (Exception e) {
            fail("Expected no exception. Got " + e);
            return;
        }

        // Overwrite artifact path to be "unknown" to make sure it returns false to overwrite
        app.artifactPath = "unknown";
        assertFalse(app.exists());
    }
}
