# Creates a Lambda for Centralized Logging to delete older indexes

// default sg for the VPC that allows for inbound communication from the vpc
variable "default_sg" {
  default = "sg-0fe4d9cdd204b9e5a"
}

// Create Role and Policy to allow Lambda access to the domain
resource "aws_iam_role" "es_cleaner_lambda" {
  name = "${var.domain}-index-retention-deleter"

  assume_role_policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Action": "sts:AssumeRole",
      "Principal": {
        "Service": "lambda.amazonaws.com"
      },
      "Effect": "Allow",
      "Sid": ""
    }
  ]
}
EOF
}

resource "aws_iam_policy" "es_http" {
  name = "${var.domain}-index-retention-deleter-access"
  description = "Allows the ES Cleaner Lambda Function to make HTTP Calls to the ElasticSearch Domain"

  policy = <<EOF
{
    "Version": "2012-10-17",
    "Statement": [
        {
            "Effect": "Allow",
            "Action": "es:*",
            "Resource": "${aws_elasticsearch_domain.es_logging_stack.arn}"
        }
    ]
}
EOF
}

resource "aws_iam_role_policy_attachment" "es_http_attach" {
  role = "${aws_iam_role.es_cleaner_lambda.name}"
  policy_arn = "${aws_iam_policy.es_http.arn}"
}

// Necessary for VPC access
resource "aws_iam_role_policy_attachment" "AWSLambdaVPCAccessExecutionRole" {
  role = "${aws_iam_role.es_cleaner_lambda.name}"
  policy_arn = "arn:aws:iam::aws:policy/service-role/AWSLambdaVPCAccessExecutionRole"
}

// Create a log group to store lambda logs in
resource "aws_cloudwatch_log_group" "index_retention_deleter" {
  name              = "/aws/lambda/${var.domain}-index-retention-deleter"
  retention_in_days = 60
}

// Create the Lambda Function that will delete old ElasticSearch indexes
resource "aws_lambda_function" "index_retention_deleter" {
  function_name = "${var.domain}-index-retention-deleter"
  s3_bucket     = "es-index-lifecycle-artifacts"
  s3_key        = "es_index_lifecycle.zip"
  handler       = "es_index_lifecycle"
  runtime       = "go1.x"
  role          = "${aws_iam_role.es_cleaner_lambda.arn}"
  timeout       = 120

  environment {
    variables = {
      ES_ENDPOINT = "https://${aws_elasticsearch_domain.es_logging_stack.endpoint}"
      RETENTION_DAYS = 15
      DRY_RUN = "false"
    }
  }

  depends_on    = ["aws_cloudwatch_log_group.index_retention_deleter"]
}

// Run on a Cron Basis every day
resource "aws_cloudwatch_event_rule" "IndexLifecycleCron" {
  name        = "${aws_lambda_function.index_retention_deleter.id}-cron"
  description = "Triggers ElasticSearch Index Deletion Lambda to trigger every day"
  # https://docs.aws.amazon.com/lambda/latest/dg/tutorial-scheduled-events-schedule-expressions.html
  schedule_expression = "rate(1 day)"
  is_enabled = true
}

# The Target for the event (run the task)
resource "aws_cloudwatch_event_target" "IndexLifecycleTarget" {
  rule = "${aws_cloudwatch_event_rule.IndexLifecycleCron.name}"
  arn  = "${aws_lambda_function.index_retention_deleter.arn}"
}

// Allow Cloudwatch Events to invoke the Lambda function
resource "aws_lambda_permission" "allow_cloudwatch" {
  statement_id  = "AllowExecutionFromCloudWatch"
  action        = "lambda:InvokeFunction"
  function_name = "${aws_lambda_function.index_retention_deleter.function_name}"
  principal     = "events.amazonaws.com"
}

