# Artifacts related to S3 Buckets for Grid

variable "allowed_roles" {
  type = "list"
  default = [
    "arn:aws:iam::425992774280:role/selenium-grid", // cape-qe
    "arn:aws:iam::148380705220:role/selenium-grid", // twilight
    "arn:aws:iam::100860887800:role/selenium-grid", // beacon
    "arn:aws:iam::147030575244:role/selenium-grid", // browsergrid-prod
  ]
}

variable "artifacts_bucket_name" {
  default = "grid-artifacts"
}

resource "aws_s3_bucket" "artifacts" {
  bucket = "${var.artifacts_bucket_name}"
  acl    = "bucket-owner-full-control"

  versioning {
    enabled = true
  }

  tags {
    Name = "Grid Artifacts"
  }

  replication_configuration {
    role = "${aws_iam_role.replication.arn}"

    rules {
      status = "Enabled"

      destination {
        bucket        = "${aws_s3_bucket.artifacts_replica.arn}"
        storage_class = "ONEZONE_IA"
      }
    }
  }
}

resource "aws_s3_bucket" "artifacts_replica" {
  provider = "aws.us-east-2"
  bucket = "${var.artifacts_bucket_name}-replica"

  versioning {
    enabled = true
  }
}

resource "aws_s3_bucket_policy" "artifacts" {
  bucket = "${aws_s3_bucket.artifacts.id}"

  policy = <<POLICY
{
    "Version": "2012-10-17",
    "Statement": [
        {
            "Effect": "Allow",
            "Principal": {
                "AWS": ${jsonencode(var.allowed_roles)}
            },
            "Action": [
                "s3:GetObject"
            ],
            "Resource": [
                "arn:aws:s3:::grid-artifacts/*",
                "arn:aws:s3:::grid-artifacts"
            ]
        },
        {
            "Effect": "Allow",
            "Principal": {
                "AWS": [
                  "arn:aws:iam::043714768218:role/jenkins-prod-master",
                  "arn:aws:iam::043714768218:role/jenkins-prod-slave"
                ]
            },
            "Action": [
                "s3:GetObject",
                "s3:PutObject",
                "s3:PutObjectAcl"
            ],
            "Resource": [
                "arn:aws:s3:::grid-artifacts/*",
                "arn:aws:s3:::grid-artifacts"
            ]
        }
    ]
}
POLICY
}

# Create an IAM Role that allows Jenkins to upload artifacts into the bucket
# Necessary for the PutObjectAcl - solely an s3 access policy wouldn't allow this
resource "aws_iam_role" "artifacts" {
  name = "jenkins_grid_artifact_upload"
  description = "Allows jenkins to upload artifacts into Grid"
  assume_role_policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Effect": "Allow",
      "Principal": {
        "AWS": [
          "arn:aws:iam::043714768218:role/jenkins-prod-master",
          "arn:aws:iam::043714768218:role/jenkins-prod-slave"
        ]
      },
      "Action": "sts:AssumeRole"
    }
  ]
}
EOF
}

resource "aws_iam_role_policy" "artifacts_upload" {
  name = "Artifacts-S3-Upload"
  role = "${aws_iam_role.artifacts.id}"

  policy = <<-EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Action": [
        "s3:GetObject",
        "s3:PutObject",
        "s3:PutObjectAcl"
      ],
      "Effect": "Allow",
      "Resource": "arn:aws:s3:::grid-artifacts/*"
    }
  ]
}
  EOF
}

# Pull a docker image that allows building chef artifacts
# https://git.xarth.tv/qe/chef_builder
resource "aws_iam_role_policy" "chef_builder_pull" {
  name = "Chef-Builder-ECR-Pull"
  role = "${aws_iam_role.artifacts.id}"

  policy = <<-EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Action": [
        "ecr:BatchCheckLayerAvailability",
        "ecr:GetDownloadUrlForLayer",
        "ecr:GetRepositoryPolicy",
        "ecr:DescribeRepositories",
        "ecr:ListImages",
        "ecr:DescribeImages",
        "ecr:BatchGetImage"
      ],
      "Effect": "Allow",
      "Resource": "arn:aws:ecr:us-west-2:147030575244:repository/twitch_chef_builder"
    },
    {
        "Sid": "AuthToken",
        "Effect": "Allow",
        "Action": "ecr:GetAuthorizationToken",
        "Resource": "*"
    }
  ]
}
  EOF
}

# Replication
resource "aws_iam_role" "replication" {
  name = "tf-iam-role-replication-${var.artifacts_bucket_name}"

  assume_role_policy = <<POLICY
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Action": "sts:AssumeRole",
      "Principal": {
        "Service": "s3.amazonaws.com"
      },
      "Effect": "Allow",
      "Sid": ""
    }
  ]
}
POLICY
}

resource "aws_iam_policy" "replication" {
  name = "tf-iam-role-policy-replication-${var.artifacts_bucket_name}"

  policy = <<POLICY
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Action": [
        "s3:GetReplicationConfiguration",
        "s3:ListBucket"
      ],
      "Effect": "Allow",
      "Resource": [
        "${aws_s3_bucket.artifacts.arn}"
      ]
    },
    {
      "Action": [
        "s3:GetObjectVersion",
        "s3:GetObjectVersionAcl"
      ],
      "Effect": "Allow",
      "Resource": [
        "${aws_s3_bucket.artifacts.arn}/*"
      ]
    },
    {
      "Action": [
        "s3:ReplicateObject",
        "s3:ReplicateDelete"
      ],
      "Effect": "Allow",
      "Resource": "${aws_s3_bucket.artifacts_replica.arn}/*"
    }
  ]
}
POLICY
}

resource "aws_iam_role_policy_attachment" "replication" {
  role       = "${aws_iam_role.replication.name}"
  policy_arn = "${aws_iam_policy.replication.arn}"
}

output "artifacts_aws_iam_role_arn" {
  value = "${aws_iam_role.artifacts.arn}"
}
