// Create a Service Role used by Beanstalk when making actions
// Creating a custom one and the default (elastic-beanstalk-service-role) is not present unless you launch via the UI
resource "aws_iam_role" "GridEBServiceRole" {
  name = "GridElasticBeanstalkServiceRole"
  assume_role_policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Sid": "",
      "Effect": "Allow",
      "Principal": {
        "Service": "elasticbeanstalk.amazonaws.com"
      },
      "Action": "sts:AssumeRole",
      "Condition": {
        "StringEquals": {
          "sts:ExternalId": "elasticbeanstalk"
        }
      }
    }
  ]
}
EOF
}

// Attach the Service Policy to the Service Role
resource "aws_iam_role_policy_attachment" "AWSElasticBeanstalkService" {
  role       = "${aws_iam_role.GridEBServiceRole.name}"
  policy_arn = "${data.aws_iam_policy.AWSElasticBeanstalkService.arn}"
}

// Create an IAM Role that allows Jenkins to manage Grid resources
resource "aws_iam_role" "artifacts" {
  name = "jenkins_grid"
  description = "Allows Jenkins to Manage Grid"
  assume_role_policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Effect": "Allow",
      "Principal": {
        "AWS": [
          "arn:aws:iam::043714768218:role/jenkins-prod-master",
          "arn:aws:iam::043714768218:role/jenkins-prod-slave"
        ]
      },
      "Action": "sts:AssumeRole"
    }
  ]
}
EOF
}

// Manage Beanstalk Versions, and fetch artifacts from the grid-artifacts bucket
resource "aws_iam_role_policy" "manage_beanstalk" {
  name = "Manage-Beanstalk"
  role = "${aws_iam_role.artifacts.id}"

  policy = <<-EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Action": [
        "elasticbeanstalk:CreateApplicationVersion",
        "elasticbeanstalk:DescribeApplicationVersions",
        "elasticbeanstalk:DeleteApplicationVersion",
        "elasticbeanstalk:UpdateApplicationVersion"
      ],
      "Effect": "Allow",
      "Resource": "arn:aws:elasticbeanstalk:${data.aws_region.current.name}:${data.aws_caller_identity.current.account_id}:applicationversion/grid-hub/*"
    },
    {
      "Action": "s3:*",
      "Effect": "Allow",
      "Resource": [
        "arn:aws:s3:::elasticbeanstalk-${data.aws_region.current.name}-${data.aws_caller_identity.current.account_id}",
        "arn:aws:s3:::elasticbeanstalk-${data.aws_region.current.name}-${data.aws_caller_identity.current.account_id}/*"
      ]
    }
  ]
}
  EOF
}

# Used for managing Node Deployments
resource "aws_iam_role" "jenkins_grid_node_deploy" {
  name = "jenkins_grid_node_deploy"
  description = "Allows Jenkins to Deploy Grid Nodes"
  assume_role_policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Effect": "Allow",
      "Principal": {
        "AWS": [
          "arn:aws:iam::043714768218:role/jenkins-prod-master",
          "arn:aws:iam::043714768218:role/jenkins-prod-slave"
        ]
      },
      "Action": "sts:AssumeRole"
    }
  ]
}
EOF
}

# Allow permissions necessary for the deploy job
resource "aws_iam_role_policy" "node_deploy" {
  name = "Node-Deploy"
  role = "${aws_iam_role.jenkins_grid_node_deploy.id}"

  policy = <<-EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Sid": "DescribeHubEnvironments",
      "Action": [
        "elasticbeanstalk:DescribeEnvironments",
        "elasticbeanstalk:DescribeConfigurationSettings",
        "elasticbeanstalk:RestartAppServer"
      ],
      "Effect": "Allow",
      "Resource": "arn:aws:elasticbeanstalk:${data.aws_region.current.name}:${data.aws_caller_identity.current.account_id}:environment/grid-hub/*"
    },
    {
      "Sid": "DeployGrid",
      "Action": [
        "autoscaling:CreateOrUpdateTags",
        "autoscaling:DescribeAutoScalingGroups",
        "autoscaling:TerminateInstanceInAutoScalingGroup"
      ],
      "Effect": "Allow",
      "Resource": "*"
    },
    {
      "Sid": "RebootApp",
      "Action": [
        "cloudformation:DescribeStackResource",
        "elasticloadbalancing:RegisterInstancesWithLoadBalancer"
      ],
      "Effect": "Allow",
      "Resource": "*"
    },
    {
      "Sid": "S3PermissionsForDescribeConfigTemplate",
      "Action": "s3:*",
      "Effect": "Allow",
      "Resource": [
        "arn:aws:s3:::elasticbeanstalk-*"
      ]
    },
    {
      "Sid": "EC2PermissionsForDescribeConfigTemplate",
      "Action": "ec2:DescribeSubnets",
      "Effect": "Allow",
      "Resource": [
        "*"
      ]
    }
  ]
}
  EOF
}

# Allow permissions necessary for the deploy job via Chef
resource "aws_iam_role_policy" "node_chef_deploy" {
  name = "Node_Chef_Deploy"
  role = "${aws_iam_role.jenkins_grid_node_deploy.id}"

  policy = <<-EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Sid": "Stmt1598387765312",
      "Action": [
        "ssm:ListCommands",
        "ssm:SendCommand"
      ],
      "Effect": "Allow",
      "Resource": "*"
    }
  ]
}
  EOF
}

# Modify Instance Type for the browser-grid-win10-modify-instance-type job
resource "aws_iam_role_policy" "node_modify_instance_type" {
  name = "Node-Modify-Instance-Type"
  role = "${aws_iam_role.jenkins_grid_node_deploy.id}"

  policy = <<-EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Sid": "NodeModifyInstanceType",
      "Action": [
        "ec2:ModifyInstanceAttribute",
        "ec2:StartInstances",
        "ec2:StopInstances",
        "ec2:DescribeInstances"
      ],
      "Effect": "Allow",
      "Resource": "*"
    }
  ]
}
  EOF
}
