variable "environment" {}
variable "subnets" {
  type = "list"
} // The subnets to place resources in
variable "vpc_id" {
  type = "string"
}

variable "elb_security_groups" {
  type = "list"
  default = [""]
}

// This is a security group Beanstalk may automatically modify to allow permissions
// DO NOT pass something like twitch_subnets to this. Beanstalk may modify that
variable "elb_managed_security_group" {
  type = "string"
  default = ""
}

variable "instance_security_groups" {
  type = "list"
  default = [""]
}

variable "instance_profile_arn" {}

variable "application_name" {}

# https://docs.aws.amazon.com/elasticbeanstalk/latest/platforms/platforms-retiring.html
variable "solution_stack_name" { default = "64bit Amazon Linux 2018.03 v2.16.11 running Docker 19.03.13-ce" }
variable "instance_type" { default = "t3.small" }

# CBG Companion Variables
variable "cbg_run_companion" { default = "true" }
variable "cbg_companion_router_url" {}
variable "cbg_companion_hub_external_port" { default = "80" }

# Environment Variables
variable "cbg_trigger_pd_events" { default = "true" }
variable "se_major_version" { default = "3" } // The major version of Selenium to use (3, 4, etc.)

variable "load_balancer_type" { // the LB to use with Hub. Se4 requires network.
  default = "classic"
}

variable "enable_se4_listeners" { // Enable Selenium 4 Listener Ports (4443, 4442). False for Se3.
  default = "false"
}

# Create a config template
resource "aws_elastic_beanstalk_configuration_template" "hub" {
  name                = "${var.application_name}-${var.environment}-config"
  application         = "${var.application_name}"
  solution_stack_name = "${var.solution_stack_name}"

  setting {
    namespace = "aws:ec2:vpc"
    name      = "VPCId"
    value     = "${var.vpc_id}"
  }

  setting {
    namespace = "aws:ec2:vpc"
    name      = "Subnets"
    value     = "${join(",", var.subnets)}"
  }

  setting {
    namespace = "aws:autoscaling:launchconfiguration"
    name      = "InstanceType"
    value     = "${var.instance_type}"
  }

  setting {
    namespace = "aws:autoscaling:asg"
    name      = "MinSize"
    value     = "1"
  }

  setting {
    namespace = "aws:autoscaling:asg"
    name      = "MaxSize"
    value     = "1"
  }

  setting {
    namespace = "aws:autoscaling:launchconfiguration"
    name      = "SecurityGroups"
    value     = "${join(",", var.instance_security_groups)}"
  }

  setting {
    namespace = "aws:autoscaling:launchconfiguration"
    name      = "IamInstanceProfile"
    value     = "${var.instance_profile_arn}"
  }

  #################
  # Load Balancer #
  #################

  setting {
    namespace = "aws:elasticbeanstalk:environment"
    name      = "LoadBalancerType"
    value     = "${var.load_balancer_type}"
  }

  setting {
    namespace = "aws:elasticbeanstalk:environment:process:4442"
    name      = "Port"
    value     = "4442"
  }

  setting {
    namespace = "aws:elasticbeanstalk:environment:process:4443"
    name      = "Port"
    value     = "4443"
  }

  setting {
    namespace = "aws:elbv2:listener:4442"
    name      = "DefaultProcess"
    value     = "4442"
  }

  setting {
    namespace = "aws:elbv2:listener:4442"
    name      = "ListenerEnabled"
    value     = "${var.enable_se4_listeners}"
  }

  setting {
    namespace = "aws:elbv2:listener:4443"
    name      = "DefaultProcess"
    value     = "4443"
  }

  setting {
    namespace = "aws:elbv2:listener:4443"
    name      = "ListenerEnabled"
    value     = "${var.enable_se4_listeners}"
  }

  setting {
    namespace = "aws:ec2:vpc"
    name      = "ELBScheme"
    value     = "internal"
  }

  setting {
    namespace = "aws:elb:loadbalancer"
    name      = "SecurityGroups"
    value     = "${join(",", var.elb_security_groups)}"
  }

  // Prevents beanstalk from creating its own Security Group for the ELB
  setting {
    namespace = "aws:elb:loadbalancer"
    name      = "ManagedSecurityGroup"
    value     = "${var.elb_managed_security_group}"
  }

  setting {
    namespace = "aws:elb:policies"
    name      = "ConnectionDrainingEnabled"
    value     = "true"
  }

  # Requests can live for up to Grid's BROWSER_TIMEOUT (300s)
  setting {
    namespace = "aws:elb:policies"
    name      = "ConnectionDrainingTimeout"
    value     = "310"
  }

  # Requests can live for up to Grid's BROWSER_TIMEOUT (300s)
  setting {
    namespace = "aws:elb:policies"
    name      = "ConnectionSettingIdleTimeout"
    value     = "310"
  }

  // Only 1 Hub Per Environment - deploy all at once
  setting {
    namespace = "aws:autoscaling:updatepolicy:rollingupdate"
    name      = "RollingUpdateEnabled"
    value     = "false"
  }

  // Only 1 Hub Per Environment - deploy all at once
  setting {
    namespace = "aws:elasticbeanstalk:command"
    name      = "DeploymentPolicy"
    value     = "AllAtOnce"
  }

  // Only 1 Hub Per Environment - deploy all at once
  setting {
    namespace = "aws:elasticbeanstalk:command"
    name      = "BatchSize"
    value     = "100"
  }

  ####################
  # Health Reporting #
  ####################
  setting {
    namespace = "aws:elasticbeanstalk:healthreporting:system"
    name      = "SystemType"
    value     = "enhanced"
  }

  setting {
    namespace = "aws:elasticbeanstalk:healthreporting:system"
    name      = "ConfigDocument"
    value     = "${file("${path.module}/application.json")}"
  }

  #######################
  # Environment Variables
  #######################
  setting {
    namespace = "aws:elasticbeanstalk:application:environment"
    name      = "TRIGGER_PAGERDUTY_EVENTS"
    value     = "${var.cbg_trigger_pd_events}"
  }
  setting {
    namespace = "aws:elasticbeanstalk:application:environment"
    name      = "GRID_NEW_SESSION_WAIT_TIMEOUT"
    value     = "60000" // 1 minute, in ms
  }

  # Grid Companion
  setting {
    namespace = "aws:elasticbeanstalk:application:environment"
    name      = "CBG_RUN_COMPANION"
    value     = "${var.cbg_run_companion}"
  }
  setting {
    namespace = "aws:elasticbeanstalk:application:environment"
    name      = "CBG_COMPANION_ROUTER_URL"
    value     = "${var.cbg_companion_router_url}"
  }
  setting {
    namespace = "aws:elasticbeanstalk:application:environment"
    name      = "CBG_COMPANION_HUB_EXTERNAL_PORT"
    value     = "${var.cbg_companion_hub_external_port}"
  }
  setting {
    namespace = "aws:elasticbeanstalk:application:environment"
    name      = "SE_MAJOR_VERSION"
    value     = "${var.se_major_version}"
  }
}

output "name" {
  value = "${aws_elastic_beanstalk_configuration_template.hub.name}"
}
