// Creates VPC Endpoint Service Connections between Grid Router and Hub

// Create the Hub VPC Endpoint Service
module "vpc_endpoint" {
  source                = "../../../vpc_endpoint_service"
  enable                = "${var.enable_vpc_endpoint}"
  environment_name      = "${var.application_name}-${var.environment}-${var.cluster_id}"
  loadbalancer_dns_name = "${aws_elastic_beanstalk_environment.hub.endpoint_url}"
  vpc_id                = "${var.vpc_id}"
  subnets               = ["${var.subnets}"]
}

// Look up the security group for Grid Router to allow Hub VPC Endpoint Connections
data "aws_security_group" "grid_router_hub_vpc_connections" {
  count    = "${var.enable_vpc_endpoint ? 1 : 0}"
  provider = "aws.twitch-cape-qe-aws"
  filter {
    name = "group-name"
    values = ["HubVPCEndpointConnections"]
  }
}

// Connect Grid Router to the Hub VPC Endpoint
resource "aws_vpc_endpoint" "hub_endpoint_for_grid_router" {
  count              = "${var.enable_vpc_endpoint ? 1 : 0}"
  provider           = "aws.twitch-cape-qe-aws"
  vpc_id             = "${var.twitch_cape_qe_aws_vpc_id}"
  service_name       = "${module.vpc_endpoint.service_name[0]}"
  vpc_endpoint_type  = "Interface"
  security_group_ids = ["${data.aws_security_group.grid_router_hub_vpc_connections.id}"]
  subnet_ids = [
    "${var.twitch_cape_qe_aws_subnets}"
  ]

  tags {
    Name = "${var.application_name}-${var.environment}-${var.cluster_id}"
  }
}

// Security Group Rules for the VPCE NLB

data "aws_network_interface" "vpce_nlb_a" {
  count = "${var.enable_vpc_endpoint ? 1 : 0}"
  filter {
    name   = "description"
    values = ["ELB ${module.vpc_endpoint.nlb_arn_suffix[0]}"]
  }
  filter {
    name   = "availability-zone"
    values = ["${data.aws_region.current.name}a"] // ex. us-west-2a
  }
}

data "aws_network_interface" "vpce_nlb_b" {
  count = "${var.enable_vpc_endpoint ? 1 : 0}"
  filter {
    name   = "description"
    values = ["ELB ${module.vpc_endpoint.nlb_arn_suffix[0]}"]
  }
  filter {
    name   = "availability-zone"
    values = ["${data.aws_region.current.name}b"] // ex. us-west-2b
  }
}

data "aws_network_interface" "vpce_nlb_c" {
  count = "${var.enable_vpc_endpoint ? 1 : 0}"
  filter {
    name   = "description"
    values = ["ELB ${module.vpc_endpoint.nlb_arn_suffix[0]}"]
  }
  filter {
    name   = "availability-zone"
    values = ["${data.aws_region.current.name}c"] // ex. us-west-2c
  }
}

// Allow any instance in the VPC to ingress with the ELB
// this is for a VPC Endpoint's NLB. The IPs may change.
resource "aws_security_group_rule" "hub_elb_vpce" { // Using this to prevent circular dependencies
  count              = "${var.enable_vpc_endpoint ? 1 : 0}"
  security_group_id = "${var.elb_managed_security_group}"
  type        = "ingress"
  description = "Connections from NLB"
  protocol    = "tcp"
  from_port   = 80
  to_port     = 80
  cidr_blocks = [
    "${data.aws_network_interface.vpce_nlb_a.private_ip}/32",
    "${data.aws_network_interface.vpce_nlb_b.private_ip}/32",
    "${data.aws_network_interface.vpce_nlb_c.private_ip}/32",
  ]
}
