<powershell>
<#
This Script is used as UserData for Windows 10 EC2 Instances.
Its responsibility is to do very initial setup, and then download an additional provision file
The provision file is external so that changes to file don't require a tear down / spin up from Terraform
(in terraform, any change to userdata will terminate existing instances)
#>

$grid_environment = "${environment}" # Capture the environment variable from Terraform
$s3_artifact_bucket = "grid-artifacts"

# Fetch the Git Branch from the AWS Tags to know where to download the artifacts from
$instanceId = (New-Object System.Net.WebClient).DownloadString("http://169.254.169.254/latest/meta-data/instance-id")
$git_branch = Get-EC2Tag | ` Where-Object {$_.ResourceId -eq $instanceId -and $_.Key -eq 'Branch'}
$git_branch_name = $($git_branch).value

Write-Host "Provisioning Machine. Environment: $grid_environment Branch: $git_branch_name"
[Environment]::SetEnvironmentVariable("GRID_ENVIRONMENT", "$grid_environment", "Machine") # Set an ENV Variable so other files can reference

if (([string]::IsNullOrEmpty($git_branch_name)))
{
  Write-Host "[ERROR] Must provide GIT_BRANCH_NAME in provision script"
  exit
}

# Main directory for storing Grid files
$config_directory = "C:\ProgramData\grid-config"
New-Item -type directory $config_directory # Create the directory

# Download the provision file
Write-Host  "Downloading Provision File"
Read-S3Object -BucketName "$s3_artifact_bucket" -File "$config_directory\resources.zip" -Key "git/$git_branch_name/resources.zip"

Expand-Archive -Path "$config_directory\resources.zip" -DestinationPath "$config_directory"

# Download the Chef Artifacts
Write-Host  "Downloading Chef Artifacts"
$chef_artifacts_file = "chef_artifacts.zip"
Read-S3Object -BucketName "$s3_artifact_bucket" -File "$config_directory\$chef_artifacts_file" -Key "git/$git_branch_name/$chef_artifacts_file"

New-Item -type directory "$config_directory\chef" # Create the directory
Expand-Archive -Path "$config_directory/$chef_artifacts_file" -DestinationPath "$config_directory\chef"
Remove-Item -Path "$config_directory\$chef_artifacts_file" # Clean Up

# Run Provisioning File - Pass it the Grid Environment and S3 URL
& "$config_directory\provision.ps1" $grid_environment $s3_url
</powershell>
