variable "cluster_id" {} // The Cluster ID used for naming resources

variable "count" {
  default = "0"
}

variable "name" {
  default = "grid-win10"
}

# Prod vs Dev
variable "environment" {
  default = "prod"
}

variable "git_branch_name" {
  default = "master"
}

variable "instance_profile" {
}

variable "instance_type" {
  default = "c4.large"
}

variable "ami" {} // the AMI to launch the instance with

variable "hub_fqdn" {
  default = "grid.us-west2.justin.tv"
}

variable "se_major_version" { // The major version of Selenium to use (3, 4, etc.)
  default = "3"
}

# The size of the volume attached.
# Setting default to 42 to account for downloaded Win Updates, etc.
variable "root_block_volume_size" {
  default = "42"
}

variable "detailed_monitoring" {
  default = "false"
}

# See https://www.terraform.io/docs/providers/aws/r/instance.html#volume_type-1 for valid inputs
variable "root_block_volume_type" {
  default = "gp2"
}

# Whether Grid binary should launch at startup
variable "AutoLaunchGrid" {
  default = "true"
}

variable "ChromeDriverVersion" {
  default = ""
}

# The node supports running tests against a desktop application
variable "DesktopAppSupport" {
  default = "false"
}

# Control which subnet the instances are brought up in
variable "subnet_id" {
  type = "list"
}

variable "security_groups" {
  type = "list"
}

variable "sns_arn_low" {
  default = ""
}
variable "sns_arn_high" {
  default = ""
}

variable "tenancy" {
  default = "default"
}

variable "asg_suspended_processes" { // https://docs.aws.amazon.com/autoscaling/ec2/userguide/as-suspend-resume-processes.html#choosing-suspend-resume
  type = "list"
  default = []
}

data "template_file" "provisioner" {
  template = "${file("${path.module}/get_provision.ps1")}"

  vars {
    environment = "${var.environment}"
  }
}

resource "aws_launch_configuration" "node" {
  name_prefix   = "${var.name}-${var.environment}-${var.cluster_id}-"
  image_id      = "${var.ami}"
  instance_type = "${var.instance_type}"
  placement_tenancy = "${var.tenancy}"
  enable_monitoring = "${var.detailed_monitoring}"

  security_groups = ["${var.security_groups}"]
  iam_instance_profile = "${var.instance_profile}"
  user_data = "${data.template_file.provisioner.rendered}"

  root_block_device {
    volume_size = "${var.root_block_volume_size}"
    volume_type = "${var.root_block_volume_type}"
  }

  lifecycle {
    create_before_destroy = true
  }
}

resource "aws_autoscaling_group" "node" {
  name = "${var.name}-${var.environment}-${var.cluster_id}"
  vpc_zone_identifier = "${var.subnet_id}"
  desired_capacity   = "${var.count}"
  max_size           = "${var.count}"
  min_size           = "${var.count}"
  launch_configuration = "${aws_launch_configuration.node.name}"
  health_check_type = "EC2"
  enabled_metrics = ["GroupInServiceInstances", "GroupPendingInstances", "GroupStandbyInstances", "GroupTerminatingInstances", "GroupTotalInstances"]
  suspended_processes = "${var.asg_suspended_processes}"
  termination_policies = ["OldestInstance"]
  wait_for_capacity_timeout = 0

  tags = [
    {
      key = "Name"
      value = "${var.name}-${var.environment}-${var.cluster_id}"
      propagate_at_launch = true
    },
    {
      key                 = "Owner"
      value               = "reichsta"
      propagate_at_launch = true
    },
    {
      key                 = "Environment"
      value               = "${var.environment}"
      propagate_at_launch = true
    },
    {
      key                 = "Branch"
      value               = "${var.git_branch_name}"
      propagate_at_launch = true
    },
    {
      key                 = "Service"
      value               = "grid"
      propagate_at_launch = true
    },
    {
      key                 = "AutoLaunchGrid"
      value               = "${var.AutoLaunchGrid}"
      propagate_at_launch = true
    },
    {
      key                 = "Hub_FQDN"
      value               = "${var.hub_fqdn}"
      propagate_at_launch = true
    },
    {
      key                 = "ChromeDriverVersion"
      value               = "${var.ChromeDriverVersion}"
      propagate_at_launch = true
    },
    {
      key                 = "DesktopAppSupport"
      value               = "${var.DesktopAppSupport}"
      propagate_at_launch = true
    },
    {
      key                 = "GridResourceType"
      value               = "node"
      propagate_at_launch = true
    },
    {
      key                 = "GridClusterID"
      value               = "${var.cluster_id}"
      propagate_at_launch = true
    },
    {
      key                 = "SeMajorVersion"
      value               = "${var.se_major_version}"
      propagate_at_launch = true
    },
  ]
}

output "asg_name" {
  value = "${aws_autoscaling_group.node.name}"
}
