package companion

import (
	"code.justin.tv/qe/grid_companion/pkg/config"
	"code.justin.tv/qe/grid_companion/pkg/services"
	"errors"
	"fmt"
	"log"
	"time"
)

// Reporter is responsible for getting the status from the hub and reporting it to Grid Router
type Reporter struct {
	Config     *config.Config
	Hub         services.Hub
	GridRouter *services.GridRouter
}

// Initializes a reporter object with default objects if they weren't already provided.
func (r *Reporter) Init(config *config.Config) {
	r.Config = config

	if r.Hub == nil {
		hubHost := fmt.Sprintf("%s:%s", *r.Config.HubInternalIP, *r.Config.HubInternalPort)
		if r.Config.GraphQL == true {
			hub := services.Hub4{}
			hub.Init("http", hubHost)
			r.Hub = &hub
		} else {
			hub := services.Hub3{}
			hub.Init("http", hubHost)
			r.Hub = &hub
		}
	}

	if r.GridRouter == nil {
		gridRouter := services.GridRouter{}
		gridRouter.Init(r.Config.GridRouterURL)
		r.GridRouter = &gridRouter
	}
}

// Infinitely will get the status from hub and report it to the grid router
func (r *Reporter) PollStatusAndReport() {
	for {
		err := r.getStatusAndReport()
		if err != nil {
			// Do not quit on error. If a dependency were to go down, we want to keep trying.
			log.Printf("WARN: Came across an error while getting hub status or reporting. The error: %v", err)
		}

		// Wait a configurable amount of seconds before the next poll
		time.Sleep(*r.Config.PollWait)
	}
}

// Gets the status from a hub and then reports that status to Grid Router
func (r *Reporter) getStatusAndReport() error {
	statusResult, err := r.Hub.Get() // Get the status from hub
	if err != nil {
		return err
	}

	if r.Config == nil {
		return errors.New("config was nil")
	}

	// Report the hub's status to Grid Router
	if r.Config.DryRun == false {
		err = r.GridRouter.ReportStatus(*r.Config, statusResult)
		if err != nil {
			return err
		}
	} else {
		log.Printf("[DryRun]: Would report: %+v", statusResult)
	}

	return nil
}
