package companion

import (
	"code.justin.tv/qe/grid_companion/pkg/config"
	"code.justin.tv/qe/grid_companion/pkg/services"
	"fmt"
	"github.com/stretchr/testify/assert"
	"net/http"
	"net/http/httptest"
	"testing"
)

func TestReporter_Init(t *testing.T) {
	testGridRouterURL := "http://localhost:1234"
	testHubID := "i-test"
	testHubIP := "localhost"
	testHubPort := "5678"
	testHubExtIP := "1.2.3.4"
	testHubExtPort := "80"

	mockConfig := config.Config{
		GridRouterURL:   &testGridRouterURL,
		HubID:           &testHubID,
		HubInternalIP:   &testHubIP,
		HubInternalPort: &testHubPort,
		HubExternalIP: 	 &testHubExtIP,
		HubExternalPort: &testHubExtPort,
	}

	mockHubURL := fmt.Sprintf("http://%s:%s%s", *mockConfig.HubInternalIP, *mockConfig.HubInternalPort, "/grid/api/hub")
	mockDefaultHub := services.Hub3{
		URL: &mockHubURL,
	}

	mockGridRouterURL := fmt.Sprintf("%s%s", *mockConfig.GridRouterURL, "/cbg/api/hub/registry")
	mockDefaultGridRouter := services.GridRouter{
		URL: &mockGridRouterURL,
	}

	mockReporter := Reporter{
		Config: &mockConfig,
		Hub: &mockDefaultHub,
		GridRouter: &mockDefaultGridRouter,
	}

	/*
	Should Initialize from Defaults when Null
	*/
	actualReporter := Reporter{}
	actualReporter.Init(&mockConfig)
	assert.Equal(t, mockReporter, actualReporter)

	/*
	Should not initialize default when provided
	 */

	 // Load in a Mock Hub3, keep the rest defaults
	 mockFakeHubUrl := "helloworld"
	 mockHub := services.Hub3{
	 	URL: &mockFakeHubUrl,
	 }
	 mockReporter = Reporter{
	 	Hub: &mockHub,
	 }

	 expectedReporter := Reporter{
	 	Config: &mockConfig,
	 	Hub: &mockHub,
	 	GridRouter: &mockDefaultGridRouter,
	 }

	 mockReporter.Init(&mockConfig)
	 assert.Equal(t, expectedReporter, mockReporter)

	 // Load in a Mock Router, keep the rest defaults
	 mockFakeRouterUrl := "http://www.notreal.com"
	 mockGridRouter := services.GridRouter{
	 	URL: &mockFakeRouterUrl,
	 }

	 expectedReporter = Reporter{
	 	Config: &mockConfig,
	 	Hub: &mockDefaultHub,
	 	GridRouter: &mockGridRouter,
	 }

	 mockReporter = Reporter{
	 	GridRouter: &mockGridRouter,
	 }
	 mockReporter.Init(&mockConfig)

	 assert.Equal(t, expectedReporter, mockReporter)
}

func TestReporter_PollStatus(t *testing.T) {
	t.Skip("Infinite Loops complex to test. Instead, testing private method getStatusAndReport()")
}

func TestReporter_getStatusAndReport(t *testing.T) {
	hubRequestCount := 0
	routerRequestCount := 0
	httpErrorMsg := "unexpected error"

	hubServer := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		hubRequestCount += 1
		fmt.Fprintf(w, "{\"slotCounts\": {\"free\": %d, \"total\": %d}}", 0, 10)
	}))
	defer hubServer.Close()

	routerServer := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		routerRequestCount += 1
		w.WriteHeader(200)
	}))
	defer routerServer.Close()

	mockHub := services.Hub3{
		URL: &hubServer.URL,
	}

	mockRouter := services.GridRouter{
		URL: &routerServer.URL,
	}

	testHubID      := "i-test"
	testHubIP      := "localhost"
	testHubPort    := "5678"
	testHubExtIP   := "1.2.3.4"
	testHubExtPort := "80"

	mockConfig := config.Config{
		GridRouterURL:   &routerServer.URL,
		HubID:           &testHubID,
		HubInternalIP:   &testHubIP,
		HubInternalPort: &testHubPort,
		HubExternalIP:   &testHubExtIP,
		HubExternalPort: &testHubExtPort,
	}

	mockReporter := Reporter{
		Hub: &mockHub,
		GridRouter: &mockRouter,
	}
	mockReporter.Init(&mockConfig)

	preHubRequestCount := hubRequestCount
	preRouterRequestCount := routerRequestCount
	err := mockReporter.getStatusAndReport()
	assert.NoError(t, err)
	assert.Equal(t, preHubRequestCount + 1, hubRequestCount, "The request count to the hub should have incremented")
	assert.Equal(t, preRouterRequestCount + 1, routerRequestCount, "The request count to the router should have incremented")

	/*
	Test when the hub server returns an error
	 */
	hubServerErr := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		hubRequestCount += 1
		http.Error(w, httpErrorMsg, 500)
	}))
	defer hubServerErr.Close()

	mockHubErr := services.Hub3{
		URL: &hubServerErr.URL,
	}
	mockReporter = Reporter{
		Hub: &mockHubErr,
		GridRouter: &mockRouter,
	}
	mockReporter.Init(&mockConfig)

	preHubRequestCount = hubRequestCount
	preRouterRequestCount = routerRequestCount
	err = mockReporter.getStatusAndReport()
	assert.Error(t, err)
	assert.EqualError(t, err, fmt.Sprintf("The hub returned an unexpected response. Status Code: 500 Body: %s\n", httpErrorMsg))
	assert.Equal(t, preHubRequestCount + 1, hubRequestCount, "The request count to the hub should have incremented")
	assert.Equal(t, preRouterRequestCount, routerRequestCount, "The request count to the router should NOT have incremented")

	/*
	Test when the router returns an error
	 */
	routerServerErr := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		routerRequestCount += 1
		http.Error(w, httpErrorMsg, 500)
	}))
	defer routerServer.Close()

	mockRouterErr := services.GridRouter{
		URL: &routerServerErr.URL,
	}
	mockReporter = Reporter{
		Hub: &mockHub,
		GridRouter: &mockRouterErr,
	}
	mockReporter.Init(&mockConfig)

	preHubRequestCount = hubRequestCount
	preRouterRequestCount = routerRequestCount
	err = mockReporter.getStatusAndReport()
	assert.Error(t, err)
	assert.EqualError(t, err, fmt.Sprintf("Unexpected Grid Router Response Code. Got Status Code: 500 - Body: %s\n", httpErrorMsg))
	assert.Equal(t, preHubRequestCount + 1, hubRequestCount, "The request count to the hub should have incremented")
	assert.Equal(t, preRouterRequestCount + 1, routerRequestCount, "The request count to the router should have incremented")

	t.Run("when dryRun is specified, does not call the server", func (t *testing.T) {
		mockReporter := Reporter{
			Hub: &mockHub,
			GridRouter: &mockRouter,
		}
		mockReporter.Init(&config.Config{
			GridRouterURL:   &routerServer.URL,
			HubID:           &testHubID,
			HubInternalIP:   &testHubIP,
			HubInternalPort: &testHubPort,
			HubExternalIP:   &testHubExtIP,
			HubExternalPort: &testHubExtPort,
			DryRun:          true,
		})

		preHubRequestCount := hubRequestCount
		preRouterRequestCount := routerRequestCount
		err := mockReporter.getStatusAndReport()
		assert.NoError(t, err)
		assert.Equal(t, preHubRequestCount + 1, hubRequestCount, "The request count to the hub should have incremented")
		assert.Equal(t, preRouterRequestCount, routerRequestCount, "The request count to the router should NOT have incremented")
	})
}
