package config

import (
	"errors"
	"flag"
	"fmt"
	"os"
	"time"
)

// Used to store companion configuration
type Config struct {
	GridRouterURL   *string
	HubID           *string
	HubInternalIP   *string
	HubInternalPort *string
	HubExternalIP   *string
	HubExternalPort *string
	HubClusterName  *string
	HubHostname     *string
	PollWait        *time.Duration
	DryRun          bool
	GraphQL         bool
}

// Initializes a configuration object by reading flags from the command line arguments
func (c *Config) Init() error {
	// Create a custom flagset - helps with testing to avoid error on multiple .Parse()'s
	flags := flag.NewFlagSet(os.Args[0], flag.ExitOnError)

	// Define all of the Flags
	idArg 			   := flags.String("id", "unknown", "The ID of the hub instance. Example: 'i-1234'")
	gridRouterURLArg   := flags.String("routerUrl", "", "The URL of the Grid Router instance. Example: 'http://localhost:8080'. Required, or provide -dryRun.")
	hubIntIPArg        := flags.String("hubIntIP", "localhost", "The IP of the Hub. Example: 'localhost'")
	hubIntPortArg      := flags.String("hubIntPort", "4444", "The port of the Hub running on the instance. Example: '4444'")
	hubExtIPArg        := flags.String("hubExtIP", "", "The external IP of the hub that can be reached. Example: '10.10.10.10'. Required")
	hubExternalPortArg := flags.String("hubExtPort", "80", "The port of the Hub outside instances should connect to. Example: '80'")
	hubClusterNameARg  := flags.String("hubClusterName", "", "The cluster name/id of the environment")
	hubURLArg          := flags.String("hubHostname", "", "The hostname of the Hub's Load Balancer.")
	pollWaitArg 	   := flags.Int("pollWait", 10, "The amount of time (in seconds) to wait in between polling Hub Status. Example: '10'")
	dryRun             := flags.Bool("dryRun", false, "Skip reporting capacity to Grid Router")
	graphQL            := flags.Bool("graphQL", false, "Poll hub capacity using GraphQL. Requires Selenium 4.0 or Greater")

	// Parse the arguments after the command
	err := flags.Parse(os.Args[1:])
	if err != nil {
		return err
	}

	// Check for required flags that were not passed
	// gridRouterURL should be provided if not a dry run
	if *gridRouterURLArg == "" && *dryRun == false {
		return errors.New(
			fmt.Sprintf("missing a required command line argument: routerUrl. Try running %s --help", os.Args[0]))
	}

	// Check for required flags that were not passed
	if *hubExtIPArg == "" {
		return errors.New(
			fmt.Sprintf("missing a required command line argument: hubExtIP. Try running %s --help", os.Args[0]))
	}

	pollDuration := time.Duration(*pollWaitArg) * time.Second

	// Load the fields into the Config Object
	c.HubID 		  = idArg
	c.GridRouterURL   = gridRouterURLArg
	c.HubInternalIP   = hubIntIPArg
	c.HubInternalPort = hubIntPortArg
	c.HubExternalIP   = hubExtIPArg
	c.HubExternalPort = hubExternalPortArg
	c.HubClusterName  = hubClusterNameARg
	c.HubHostname     = hubURLArg
	c.PollWait 		  = &pollDuration
	c.DryRun          = *dryRun
	c.GraphQL         = *graphQL

	return nil
}
