package config

import (
	"fmt"
	"github.com/stretchr/testify/assert"
	"os"
	"testing"
	"time"
)

func TestConfig_Init(t *testing.T) {
	// Store the original arguments and reset at the end of the test (it is global)
	oldArgs := os.Args
	defer func() { os.Args = oldArgs }()

	/*
	Test providing all arguments
	 */

	testID := "i-test"
	testRouterURL := "http://test.com"
	testHubIntIP := "localhost"
	testHubIntPort := "1234"
	testHubExtIP := "1.2.3.4"
	testHubExtPort := "5678"
	testHubClusterName := "t3stClusterName"
	testHubHost := "grid-hub.com"
	testPollWaitSeconds := 99
	testPollWait := time.Duration(testPollWaitSeconds) * time.Second
	testDryRun := true
	testGQL    := true

	expectedConfig := Config{
		HubID:           &testID,
		GridRouterURL:   &testRouterURL,
		HubInternalIP:   &testHubIntIP,
		HubInternalPort: &testHubIntPort,
		HubExternalIP:   &testHubExtIP,
		HubExternalPort: &testHubExtPort,
		HubClusterName:  &testHubClusterName,
		HubHostname:     &testHubHost,
		PollWait:        &testPollWait,
		DryRun:          testDryRun,
		GraphQL:         testGQL,
	}

	os.Args = []string{"cmd",
		fmt.Sprintf("-id=%s", testID),
		fmt.Sprintf("-routerUrl=%s",      testRouterURL),
		fmt.Sprintf("-hubIntIP=%s",       testHubIntIP),
		fmt.Sprintf("-hubIntPort=%s",     testHubIntPort),
		fmt.Sprintf("-hubExtIP=%s",       testHubExtIP),
		fmt.Sprintf("-hubExtPort=%s",     testHubExtPort),
		fmt.Sprintf("-hubClusterName=%s", testHubClusterName),
		fmt.Sprintf("-hubHostname=%s",    testHubHost),
		fmt.Sprintf("-pollWait=%d",       testPollWaitSeconds),
		fmt.Sprintf("-dryRun=%t",         testDryRun),
		"-graphQL",
	}

	actualConfig := Config{}

	err := actualConfig.Init()
	assert.NoError(t, err)

	assert.Equal(t, expectedConfig, actualConfig)

	/*
	Test missing one required argument but not the other
	 */
	// Hub Ext IP Required
	os.Args = []string{"cmd",
		fmt.Sprintf("-routerUrl=%s", testRouterURL),
	}
	actualConfig = Config{}
	err = actualConfig.Init()
	assert.Error(t, err, "missing a required command line argument")

	t.Run("-routerURL", func (t *testing.T) {
		t.Run("required without -dryRun", func (t *testing.T) {
			os.Args = []string{"cmd",
				fmt.Sprintf("-hubExtIP=%s", testHubExtIP),
			}
			actualConfig = Config{}
			err = actualConfig.Init()
			assert.Error(t, err, "missing a required command line argument")
		})

		t.Run("not required with -dryRun", func (t *testing.T) {
			os.Args = []string{"cmd",
				"-dryRun",
				fmt.Sprintf("-hubExtIP=%s", testHubExtIP),
			}
			actualConfig = Config{}
			err = actualConfig.Init()
			assert.NoError(t, err)
		})
	})

	/*
	Test providing no arguments
	*/

	os.Args = []string{"cmd"}
	actualConfig = Config{}
	err = actualConfig.Init()
	assert.Error(t, err, "missing a required command line argument")

	/*
	Test that defaults get set
	*/
	os.Args = []string{"cmd",
		fmt.Sprintf("-routerUrl=%s", testRouterURL),
		fmt.Sprintf("-hubExtIP=%s", testHubExtIP),
	}

	defaultID := "unknown"
	defaultHubIP := "localhost"
	defaultHubPort := "4444"
	defaultHubExtPort := "80"
	defaultHubClustername := ""
	defaultHubHostname := ""
	defaultPollWaitSeconds := 10
	defaultPollWait := time.Duration(defaultPollWaitSeconds) * time.Second
	defaultDryRun := false
	defaultGQL    := false

	expectedConfig = Config{
		GridRouterURL:   &testRouterURL,
		HubID:           &defaultID,
		HubInternalIP:   &defaultHubIP,
		HubInternalPort: &defaultHubPort,
		HubExternalIP:   &testHubExtIP,
		HubExternalPort: &defaultHubExtPort,
		HubClusterName:  &defaultHubClustername,
		HubHostname:     &defaultHubHostname,
		PollWait:        &defaultPollWait,
		DryRun:          defaultDryRun,
		GraphQL:         defaultGQL,
	}

	actualConfig = Config{}
	err = actualConfig.Init()
	assert.NoError(t, err)
	assert.Equal(t, expectedConfig, actualConfig)
}
