package services

import (
	"bytes"
	"code.justin.tv/qe/grid_companion/pkg/config"
	"encoding/json"
	"errors"
	"fmt"
	"io/ioutil"
	"net/http"
	"time"
)

// Provides a service connection to the Grid Router
type GridRouter struct {
	URL *string
}

// An object to send to the Grid Router
type HubStatusReporterInput struct {
	ID          *string
	IP          *string
	Port        *string
	ClusterName *string `json:"clusterName"`
	Hostname    *string
	SlotCounts  SlotCounts
}

const gridRouterReportStatusEndpoint = "/cbg/api/hub/registry"

// Initializes a Grid Router connection
func (r *GridRouter) Init(baseURL *string) {
	fullUrl := fmt.Sprintf("%s%s", *baseURL, gridRouterReportStatusEndpoint)
	r.URL = &fullUrl
}

// Reports the status of a hub to the Grid Router
func (r *GridRouter) ReportStatus(config config.Config, status *HubOutput) error {
	if status == nil {
		return errors.New("received nil as a required parameter")
	}

	// Load the data into a struct to send out
	data := HubStatusReporterInput{
		ID:          config.HubID,
		IP:          config.HubExternalIP,
		Port:        config.HubExternalPort,
		ClusterName: config.HubClusterName,
		Hostname:    config.HubHostname,
		SlotCounts:  status.SlotCounts,
	}

	// Transform the struct into a JSON String
	postBodyData, err := json.Marshal(data)
	if err != nil {
		return err
	}

	// Create the Request
	// Ex: POST http://localhost:8080/cbg/api/hub/registry/i-1234
	url := fmt.Sprintf("%s/%s", *r.URL, *config.HubID)
	req, err := http.NewRequest(http.MethodPost, url, bytes.NewBuffer(postBodyData))
	if err != nil {
		return err
	}
	req.Header.Set("Content-Type", "application/json")

	// Create the Client
	client := &http.Client{}
	client.Timeout = time.Second * 3 // We want to be speedy. Time out after 3 seconds

	// Send the Request
	fmt.Printf("-> POST %s | (Body: %s)\n", url, string(postBodyData))
	resp, err := client.Do(req)
	if err != nil {
		return err
	}
	fmt.Printf("<- [%d] %s\n", resp.StatusCode, url)

	if !(resp.StatusCode >= 200 && resp.StatusCode <= 299) {
		defer resp.Body.Close()
		body, _ := ioutil.ReadAll(resp.Body)

		return errors.New(
			fmt.Sprintf("Unexpected Grid Router Response Code. Got Status Code: %d - Body: %s", resp.StatusCode, body))
	}

	return nil
}
