package services

import (
	"code.justin.tv/qe/grid_companion/pkg/config"
	"fmt"
	"github.com/stretchr/testify/assert"
	"net/http"
	"net/http/httptest"
	"testing"
)

func TestGridRouter_Init(t *testing.T) {
	/*
	Ensure it can initialize
	 */

	expectedBase := "http://www.test.com"
	expectedUrl := fmt.Sprintf("%s%s", expectedBase, gridRouterReportStatusEndpoint)
	expected := GridRouter{
		URL: &expectedUrl,
	}

	actual := GridRouter{}
	actual.Init(&expectedBase)
	assert.Equal(t, expected, actual)
}

func TestGridRouter_ReportStatus(t *testing.T) {
	/*
	Test Happy Condition
	 */
	 testHubID := "i-test"
	 testHubIP := "localhost"
	 testHubExtPort := "80"
	 testClusterName := "t3stCluster"
	 mockConfig := config.Config{
	 	HubID: &testHubID,
	 	HubExternalIP: &testHubIP,
	 	HubExternalPort: &testHubExtPort,
	 	HubClusterName: &testClusterName,
	 }

	apiStub := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(200)
	}))
	defer apiStub.Close()

	gridRouter := GridRouter{
		URL: &apiStub.URL,
	}

	mockHubOutput := HubOutput{
		SlotCounts: SlotCounts{Free: 0, Total: 10},
	}

	err := gridRouter.ReportStatus(mockConfig, &mockHubOutput)
	assert.NoError(t, err)

	/*
	When an error is returned from the server, it should be passed to the client
	 */
	errMsg := "unexpected error"
	errCode := 500
	apiStubErr := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		http.Error(w, errMsg, errCode)
	}))
	defer apiStubErr.Close()

	gridRouter = GridRouter{
		URL: &apiStubErr.URL,
	}
	err = gridRouter.ReportStatus(mockConfig, &mockHubOutput)
	assert.Error(t, err)
	assert.EqualError(t, err, fmt.Sprintf(
		"Unexpected Grid Router Response Code. Got Status Code: %d - Body: %s\n", errCode, errMsg))

	/*
	When passed nil, it should return an error
	 */
	 mockConfig = config.Config{}
	 err = gridRouter.ReportStatus(mockConfig, nil)
	 assert.Error(t, err)
	 assert.EqualError(t, err, "received nil as a required parameter")
}
