package services

import (
	"encoding/json"
	"errors"
	"fmt"
	"io/ioutil"
	"net/http"
)

const hubEndpoint = "/grid/api/hub"

// Provides a service connection to the Hub
type Hub3 struct {
	URL *string
}

// Initializes a Hub connection
func (h *Hub3) Init(protocol string, baseURL string) {
	fullUrl := fmt.Sprintf("%s://%s%s", protocol, baseURL, hubEndpoint)
	h.URL = &fullUrl
}

// Gets the status from the Hub API
func (h *Hub3) Get() (*HubOutput, error) {
	var hubStatus HubOutput

	// If the URL wasn't set, return
	if h.URL == nil || *h.URL == "" {
		return nil, errors.New("the Hub URL was not set")
	}

	// Make a request to the Hub
	resp, err := http.Get(*h.URL)
	if err != nil {
		return nil, err
	}

	// Read the body
	defer resp.Body.Close()
	body, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		return nil, err
	}

	// Ensure the status code was valid
	if !(resp.StatusCode >= 200 && resp.StatusCode <= 299) {
		return nil, errors.New(
			fmt.Sprintf("The hub returned an unexpected response. Status Code: %d Body: %s", resp.StatusCode, body))
	}

	// Unmarshall the results into the struct
	err = json.Unmarshal(body, &hubStatus)
	if err != nil {
		return nil, err
	}

	return &hubStatus, nil
}
