package main

import (
  "bytes"
  "code.justin.tv/systems/sandstorm/manager"
  "encoding/json"
  "errors"
  "fmt"
  "github.com/PagerDuty/go-pagerduty"
  "github.com/aws/aws-lambda-go/lambda"
  "io"
  "io/ioutil"
  "log"
  "net/http"
  "strings"
  "time"
)

var baseHost = "api-prod.browsergrid.xarth.tv"

func main() {
  lambda.Start(runHealthCheck) // Comment out if running locally
  //runHealthCheck() // Uncomment if running locally
}

// Runs the health check
func runHealthCheck() error {
  config := NewConfig()
  if config == nil {
    return errors.New("configuration was nil")
  }

  // Timeout after 30 seconds
  timeout := time.Duration(30 * time.Second)

  // Create the client with that timeout value
  client := &http.Client{
    Timeout: timeout,
  }

  // Create a test session against the browser
  sid, err := createSession(client, config.Browser)
  if err != nil {
    triggerAlert(buildErrorString(err, config.Browser))
    return err
  }

  // Delete the test session
  err = deleteSession(client, sid)
  if err != nil {
    triggerAlert(buildErrorString(err, config.Browser))
    return err
  }

  return nil
}

// Creates a session on Grid
func createSession(client *http.Client, browserName string) (string, error) {
  path := "wd/hub/session"
  fullUrl := fmt.Sprintf("https://%s/%s", baseHost, path)
  strBody := fmt.Sprintf("{\"desiredCapabilities\": {\"browserName\": \"%s\"}}", browserName)
  postBody := bytes.NewBuffer([]byte(strBody)) // Convert to a buffer

  log.Printf("Creating New Session for Browser: %s", browserName)

  // Make the HTTP Request
  log.Printf("-> [POST] %s | Body: %s\n", fullUrl, strBody)
  req, err := createRequest(http.MethodPost, fullUrl, postBody)
  if err != nil {
    return "", err
  }

  resp, err := client.Do(req)
  if err != nil {
    return "", err // Return
  }
  log.Printf("<- %d\n", resp.StatusCode)

  err = checkRequestStatus(resp)
  if err != nil {
    return "", err
  }

  // Open the Body
  defer resp.Body.Close()
  respBody, err := ioutil.ReadAll(resp.Body)
  if err != nil {
    return "", err
  }

  sid, err := getSessionID(respBody)
  if err != nil {
    return "", err
  }

  return sid, nil
}

// Deletes a session on Grid
func deleteSession(client *http.Client, sessionID string) error {
  deletePath := fmt.Sprintf("wd/hub/session/%s", sessionID)
  fullUrl := fmt.Sprintf("https://%s/%s", baseHost, deletePath)

  log.Printf("-> [DELETE] %s\n", fullUrl)
  req, err := createRequest(http.MethodDelete, fullUrl, nil)
  if err != nil {
    return err
  }

  resp, err := client.Do(req)
  if err != nil {
    return err
  }
  log.Printf("<- %d\n", resp.StatusCode)
  return checkRequestStatus(resp)
}

// Checks that the request's response status is healthy. Triggers a PD Incident if not.
func checkRequestStatus(resp *http.Response) error {
  // If the response code was a success (2xx)
  if resp.StatusCode >= 200 && resp.StatusCode <= 299 {
    log.Printf("HTTP Status Ok! Returned Status Code %d", resp.StatusCode)
    return nil
  } else { // If the response code was a failure (non 2xx)
    defer resp.Body.Close()
    bodyBytes, _ := ioutil.ReadAll(resp.Body) // Return the response body in the error
    return errors.New(fmt.Sprintf("Unexpected Status Code: %d. Response Body: %s", resp.StatusCode, bodyBytes))
  }
}

// Represents a New Session Response. Used for unmarshalling the JSON BOdy
type newSessionResponse struct {
  SessionID string
  Value     struct {
    SessionID string
  }
}

// Returns the session id from a body
func getSessionID(body []byte) (string, error) {
  var sessionID string

  sessionResponse := newSessionResponse{} // Create object to store the json results

  // Unmarshal the json body into the sesionResponse object
  err := json.Unmarshal(body, &sessionResponse)
  if err != nil {
    return "", err
  }

  // Based on W3C Spec Compliance, SessionID may be in two different places
  // { "sessionID":"mySID" } or { "value": { "sessionID":"mySID" }}
  // Chromedriver does not adhere to W3C Spec, thus we need to check both spots
  if sessionResponse.Value.SessionID != "" { // If not blank, assign value to sessionID var
    sessionID = sessionResponse.Value.SessionID
  } else if sessionResponse.SessionID != "" { // If not blank, assign value to sessionID var
    sessionID = sessionResponse.SessionID
  } else {
    log.Printf("ERROR! The Session ID was found to be blank. The body that was passed to this session: %s", body)
    return "", errors.New("error! The Session ID was found to be blank")
  }

  return sessionID, nil
}

// Creates a request with the basic authentication supported
func createRequest(method string, url string, body io.Reader) (*http.Request, error) {
  gridUser := "healthcheck"
  gridPassword := getGridCredential()

  req, err := http.NewRequest(method, url, body)
  if err != nil {
    return nil, err
  }

  req.SetBasicAuth(gridUser, gridPassword)
  req.Header.Set("Content-Type", "application/json")

  return req, nil
}

func buildErrorString(err error, browserName string) string {
  return fmt.Sprintf("Grid Health Check Failed! Host: %s Browser: %s Returned Error: %v", baseHost, browserName, err)
}

func triggerAlert(description string) {
  serviceKey := "c4cf6b56f21f497aab0f596291c6e74a"

  log.Println("ALERT: " + description)
  log.Println("Sending PagerDuty event! Description:\n\n--------\n" + description + "\n--------")

  incident := pagerduty.Event{ServiceKey: serviceKey, Type: "trigger", Description: description}
  resp, err := pagerduty.CreateEvent(incident)

  if err != nil {
    log.Fatal(err)
  }

  log.Print("PagerDuty Response: " + resp.Status + " | " + resp.Message + " | " + resp.IncidentKey)
}

// @return [String] A password that will allow for access to Grid
func getGridCredential() string {
  roleArn := "arn:aws:iam::734326455073:role/sandstorm/production/templated/role/grid_router"
  secretName := "qa-eng/grid-router/production/api_access_key"

  // Set up the manager
  m := manager.New(manager.Config{
    AWSConfig: manager.AWSConfig(nil, roleArn),
  })

  secretContent, err := m.Get(secretName)
  if err != nil {
    log.Fatal(err)
  }

  secretString := string(secretContent.Plaintext[:]) // Convert from byte array to string
  secretArray := strings.Split(secretString, ",") // Each access token is comma separated.
  if len(secretArray) < 1 {
    log.Fatal("Problem Fetching Secret")
  }
  firstSecret := secretArray[0] // Grab the first access token from the array

  if len(firstSecret) <= 0 {
    log.Fatal("Problem Fetching Secret")
  }

  return firstSecret
}
