resource "aws_lambda_function" "grid_healthcheck" {
  filename         = "../main.zip"
  function_name    = "grid-healthcheck"
  role             = "${aws_iam_role.iam_for_lambda.arn}"
  handler          = "main"
  source_code_hash = "${base64sha256(file("../main.zip"))}"
  runtime          = "go1.x"
  timeout          = "30"

  vpc_config {
    security_group_ids = ["${data.terraform_remote_state.vpc.twitch_subnets_sg}"] # Allow connection from all twitch infra
    subnet_ids = ["${data.terraform_remote_state.vpc.private_a}"] # Run on the private subnet
  }

  environment {
    variables = {
      BROWSER = "chrome"
    }
  }

  tags {
    Name = "Grid Healthcheck Chrome"
  }
}

############
# Triggers #
############

# The trigger, to fire every minute
resource "aws_cloudwatch_event_rule" "every_minute" {
  name = "grid-healthcheck-every-minute"
  description = "Health Check for Grid, Fires every 1 minute"
  schedule_expression = "rate(1 minute)"
}

# Connects the trigger to the lambda fucntion
resource "aws_cloudwatch_event_target" "check_health" {
  rule = "${aws_cloudwatch_event_rule.every_minute.name}"
  arn = "${aws_lambda_function.grid_healthcheck.arn}"
}

# Gives lambda permission
resource "aws_lambda_permission" "allow_cloudwatch_to_call_check_health" {
  statement_id = "AllowExecutionFromCloudWatch"
  action = "lambda:InvokeFunction" // Enable the event
  //  action = "lambda:DisableInvokeFunction" // Disable the event
  function_name = "${aws_lambda_function.grid_healthcheck.function_name}"
  principal = "events.amazonaws.com"
  source_arn = "${aws_cloudwatch_event_rule.every_minute.arn}"
}

# Stores the results of the health check in a log
resource "aws_cloudwatch_log_group" "healthcheck_chrome" {
  name = "/aws/lambda/${aws_lambda_function.grid_healthcheck.function_name}"
  retention_in_days = "3"
}
