# Terraform Resources for the Health Check Lambda Function

/*
There are a few pieces to this:
1) A lambda function, which executes the code serverless
2) A log group, which stores the results
3) A policy for the lambda function to: 1) Put the results in a Log Group, 2) Access the VPC
4) A trigger, which will trigger it every x minutes
*/


# Refer to twitch-cape-qe-aws to be able to get security groups and subnets
data "terraform_remote_state" "vpc" {
  backend = "s3"
  config {
    profile = "twitch-cape-qe-aws"
    bucket  = "tf-state-qe"
    key     = "tfstate/qe/terraform/vpc"
    region  = "us-west-2"
  }
}

############
# Policies #
############


# An IAM Role for the Lambda Function to use
resource "aws_iam_role" "iam_for_lambda" {
  name = "grid-healthcheck-lambda"

  assume_role_policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Action": "sts:AssumeRole",
      "Principal": {
        "Service": "lambda.amazonaws.com"
      },
      "Effect": "Allow",
      "Sid": ""
    }
  ]
}
EOF
}

/*
A policy, which defines:
1) Can create log streams and put events within the pre-existing Log Group. Without this specification, it would create its own
2) Utilize EC2 resources so that the VPC Can be accessed. This lets us hit the internal website
*/
resource "aws_iam_role_policy" "main-policy" {
  name = "grid-healthcheck-lambda-policy"
  role = "${aws_iam_role.iam_for_lambda.id}"

  policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Effect": "Allow",
      "Action": [
        "logs:CreateLogStream",
        "logs:PutLogEvents"
      ],
      "Resource": [
        "${aws_cloudwatch_log_group.healthcheck_chrome.arn}",
        "${aws_cloudwatch_log_group.healthcheck_firefox.arn}"
      ]
    },
    {
      "Effect": "Allow",
      "Action": [
        "ec2:CreateNetworkInterface",
        "ec2:DescribeNetworkInterfaces",
        "ec2:DeleteNetworkInterface"
      ],
      "Resource": "*"
    }
  ]
}
EOF
}

# policy that allows your role to assume your sandstorm role
data "aws_iam_policy_document" "sandstorm" {
  statement {
    actions = [
      "sts:AssumeRole",
    ]
    resources = [
      "arn:aws:iam::734326455073:role/sandstorm/production/templated/role/grid_router"
    ]
  }
}

resource "aws_iam_role_policy" "sandstorm_policy" {
  name = "grid-healthcheck-sandstorm-policy"
  role = "${aws_iam_role.iam_for_lambda.id}"
  policy = "${data.aws_iam_policy_document.sandstorm.json}"
}
