package main

import (
	"fmt"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/elasticbeanstalk"
	"log"
)

// Interface for working with the ElasticBeanstalk API (helps for mocking)
type BeanstalkInterface interface {
	RestartAppServer(input *elasticbeanstalk.RestartAppServerInput) (*elasticbeanstalk.RestartAppServerOutput, error)
}

// Used for interacting with Elasticbeanstalk API
type BeanstalkService struct {
	Service BeanstalkInterface
	session *session.Session
}

// Returns an Elastic Beanstalk Service Handler
func NewBeanstalkService() (*BeanstalkService, error) {
	bs := &BeanstalkService{}

	sess, err := session.NewSession(&aws.Config{
		Region: aws.String("us-west-2"),
	})
	if err != nil {
		return bs, err
	}
	bs.session = sess

	//Create a ElasticBeanstalk client the session.
	svc := elasticbeanstalk.New(sess)
	bs.Service = svc
	return bs, nil
}

// Restarts a Hub's App Server on Elastic Beanstalk
// environment = the environment name of the hub
// clusterName = the cluster name of the hub
func (bs *BeanstalkService) RestartHub(environment string, clusterName string) error {
	hubAppServerName := fmt.Sprintf("grid-hub-%s-%s", environment, clusterName)
	log.Printf("Restarting Hub App Server: %s", hubAppServerName)
	_, err := bs.Service.RestartAppServer(&elasticbeanstalk.RestartAppServerInput{
		EnvironmentName: aws.String(hubAppServerName),
	})
	return err
}
