package main

import (
	"code.justin.tv/qe/grid_reboot/pkg/config"
	"code.justin.tv/qe/grid_reboot/pkg/ec2"
	"code.justin.tv/qe/grid_reboot/pkg/grid_reboot"
	"code.justin.tv/qe/grid_router/src/pkg/hub_registry"
	GridRouterSDK "code.justin.tv/qe/grid_router/src/pkg/sdk"
	"errors"
	"fmt"
	"github.com/aws/aws-sdk-go/aws"
	"log"
	"time"
)

// Error Messages
const (
	errHubAlreadyPaused = "hub was already paused"
	hubRebootSleepTime = time.Second * 10 // the time to wait to reboot the hub after rebooting the nodes

	// Total time to wait after rebooting
	// subtract the amount of time we waited before rebooting hub
	rebootSleepTime = (time.Minute * 2) - hubRebootSleepTime
)

func main() {
	//lambda.Start(runReboot) // Comment out if running locally
	runScript() // Uncomment if running locally
}

func runScript() {
	var gridConfig = config.GridRebootConfig{}
	err := gridConfig.Init()
	if err != nil {
		log.Fatalf("Problem initializing configuration. Error: %v", err)
	}

	// Set up necessary services
	grClient := GridRouterSDK.NewClient(gridConfig.GridRouterHost)

	ec2Service, err := ec2.New(gridConfig.EC2.Region)
	if err != nil {
		log.Fatalf("Problem creating EC2 Service: %v", err)
	}

	bsService, err := NewBeanstalkService()
	if err != nil {
		log.Fatalf("Problem creating Beanstalk Service: %v", err)
	}

	// Fetch all clusters in the account
	clusterIDs, err := FetchClusterIDs(gridConfig, ec2Service)
	if err != nil {
		log.Fatalf("Problem fetching clusters. Error: %v", err)
	}

	log.Printf("All returned clusters: %v", clusterIDs)

	// Process each cluster
	err = ProcessClusters(gridConfig, grClient, ec2Service, clusterIDs, bsService)
	if err != nil {
		log.Fatalf("Encountered error processing clusters: %v", err)
	}
}

// Interface for Grid Router SDK
type GridRouterClient interface {
	GetHubByClusterName(clusterName string) (*hub_registry.Hub, error)
	PauseHubByClusterName(clusterName string) error
	UnpauseHubByClusterName(clusterName string) error
}

// Returns all of the Cluster IDs within the AWS Account
// gridConfig The configuration file for grid reboot
// ec2Service the AWS EC2 Service used to query
func FetchClusterIDs(gridConfig config.GridRebootConfig, ec2Service *ec2.Handler) ([]string, error) {
	clusterIDs := make([]string, 0)

	hubs, err := grid_reboot.GetHubs(gridConfig, ec2Service)
	if err != nil { return clusterIDs, err }

	for _, hub := range hubs {
		for _, tag := range hub.Tags {
			if aws.StringValue(tag.Key) == "GridClusterID" {
				clusterIDs = append(clusterIDs, aws.StringValue(tag.Value))
				break
			}
		}
	}
	return clusterIDs, nil
}

// Will go through each cluster provided - drain & reboot them
// gridConfig The configuration file for grid reboot
// grClient The Grid Router SDK Client
// ec2Service the AWS EC2 Service used to query
func ProcessClusters(config config.GridRebootConfig, grClient GridRouterClient, ec2Service *ec2.Handler,
	clusterIDs []string, bsService *BeanstalkService) error {
	for _, clusterName := range clusterIDs {
		err := ProcessCluster(config, grClient, ec2Service, clusterName, bsService)
		if err != nil {
			if err.Error() == errHubAlreadyPaused { // if hub was already paused, just move on. May be paused for maintenance.
				continue
			}
			return err
		}
	}
	return nil
}

// Drains and reboots a specific cluster
// gridConfig The configuration file for grid reboot
// grClient The Grid Router SDK Client
// ec2Service the AWS EC2 Service used to query
func ProcessCluster(config config.GridRebootConfig, grClient GridRouterClient, ec2Service *ec2.Handler,
	clusterName string, bsService *BeanstalkService) error {
	log.Printf("Processing Cluster: %s", clusterName)

	hubEnvironment := config.EC2.Hub.Tags["Environment"]
	if len(hubEnvironment) <= 0 {
		return errors.New("unknown hub environment within config tags")
	}

	resp, err := grClient.GetHubByClusterName(clusterName)
	if err != nil { return err }

	// We don't want to modify an already paused hub. It may have been intentionally paused for maintenance.
	if resp.Paused == true {
		return errors.New(errHubAlreadyPaused)
	}

	err = grClient.PauseHubByClusterName(clusterName)
	if err != nil {
		grClient.UnpauseHubByClusterName(clusterName)
		return err
	}

	// Wait for Drain
	err = WaitForDrain(config, grClient, clusterName)
	if err != nil {
		grClient.UnpauseHubByClusterName(clusterName)
		return err
	}

	// One last check to ensure it's paused before rebooting
	paused, err := IsHubPaused(grClient, clusterName)
	if err != nil {
		// unpause just incase to not leave the cluster in a bad state
		_ = grClient.UnpauseHubByClusterName(clusterName)
		return err
	}
	if paused == false {
		// unpause just incase to not leave the cluster in a bad state. Perhaps it was a bug in getting the hub state
		_ = grClient.UnpauseHubByClusterName(clusterName)
		return errors.New("hub was not paused when it should have been")
	}

	// Reboot Nodes
	log.Printf("Rebooting %s", clusterName)
	err = grid_reboot.RebootNodesByCluster(clusterName, ec2Service, &config)
	if err != nil {
		config.Clock.Sleep(rebootSleepTime) // wait just incase they did still reboot despite the error
		grClient.UnpauseHubByClusterName(clusterName)
		return err
	}

	// Restart Hub App Server
	config.Clock.Sleep(hubRebootSleepTime)
	err = bsService.RestartHub(hubEnvironment, clusterName)
	if err != nil {
		log.Printf("WARN: encountered error restarting app server: %v\nAllowing nodes to finish rebooting.", err)
		config.Clock.Sleep(rebootSleepTime) // wait just incase they did still reboot despite the error
		grClient.UnpauseHubByClusterName(clusterName)
		return err
	}

	// Wait -- TODO: Dynamic
	config.Clock.Sleep(rebootSleepTime)

	// Unpause
	err = grClient.UnpauseHubByClusterName(clusterName)
	if err != nil { return err }

	return nil
}

// Blocking function that waits for a Hub to be drained of session requests
// Returns an error if the timeout is reached
// gridConfig The configuration file for grid reboot
// grClient The Grid Router SDK Client
// clusterName The cluster to track for draining
func WaitForDrain(config config.GridRebootConfig, grClient GridRouterClient, clusterName string) error {
	timeout := config.Clock.Now().Add(time.Minute * 15)
	drained := false
	for config.Clock.Now().Before(timeout) {
		hub, err := grClient.GetHubByClusterName(clusterName)
		if err != nil {
			log.Printf("WARN: Encountered err: %v", err)
			config.Clock.Sleep(time.Second * 5)
			continue
		}

		log.Printf("[%s] Hub Capacity: Free: %d Total: %d", clusterName, hub.SlotCounts.Free, hub.SlotCounts.Total)

		if hub.Drained() {
			log.Printf("[%s] Hub drained", clusterName)
			drained = true
			break
		}

		log.Printf("[%s] Retrying while waiting for drain...", clusterName)
		config.Clock.Sleep(time.Second * 5)
	}

	if !drained {
		return fmt.Errorf("timeout reached waiting for cluster [%s] to drain", clusterName)
	}
	return nil
}

// Returns a boolean based on if the Cluster's Hub is paused
func IsHubPaused(grClient GridRouterClient, clusterName string) (bool, error) {
	hub, err := grClient.GetHubByClusterName(clusterName)
	if err != nil {
		return false, err
	}
	return hub.Paused, nil
}
