package config

import (
	"errors"
	"fmt"
	"github.com/jonboulle/clockwork"
	"gopkg.in/yaml.v2"
	"io/ioutil"
	"log"
	"os"
)

const envName_GridRouterHost      = "GRID_ROUTER_HOST"
const envName_GridHubEnvironment  = "GRID_HUB_ENVIRONMENT"

// Contains keys and values of configuration for this application
type GridRebootConfig struct {
	EC2            EC2      `yaml:"ec2"`
	GridRouterHost string
	Clock          clockwork.Clock
	rawData        []byte
}

// Contains keys and values of configuration for specifically EC2
type EC2 struct {
	Nodes  Node   `yaml:"nodes"`
	Hub    Hub    `yaml:"hub"`
	Region string `yaml:"region"`
}

// Contains a node object from the config
type Node struct {
	Tags               map[string]string `yaml:"tags,omitempty"`
}

// Contains hub objects from the config
type Hub struct {
	Tags map[string]string `yaml:"tags,omitempty"`
}

/*
  Initializes a GridRebootConfig struct by populating results from a config file, configs/config.yaml
 */
func (c *GridRebootConfig) Init() error {
	// Load the Config
	if c.rawData == nil {
		c.LoadConfig()
	}
	err := c.populateStructFromConfigData()
	if err != nil {
		return errors.New(fmt.Sprintf("Problem unmarshalling the yaml config file: %v", err))
	}

	// Load the environment from environment variable
	gridEnvironmentENV := os.Getenv("GRID_ENVIRONMENT")
	if len(gridEnvironmentENV) > 0 {
		log.Printf("Loading environment tag from environment variable. New Value: %s", gridEnvironmentENV)
		c.EC2.Nodes.Tags["Environment"] = gridEnvironmentENV
	}

	// Load the Hub Environment from the environment variable
	gridHubEnvironmentENV := os.Getenv(envName_GridHubEnvironment)
	if len(gridHubEnvironmentENV) > 0 {
		log.Printf("Loading hub environment tag from environment variable. New Value: %s", gridHubEnvironmentENV)
		c.EC2.Hub.Tags["Environment"] = gridHubEnvironmentENV
	}

	// Load the Grid Router host from environment variable
	gridRouterHostENV := os.Getenv(envName_GridRouterHost)
	if len(gridRouterHostENV) <= 0 {
		return fmt.Errorf("required env variable %s not provided", envName_GridRouterHost)
	}
	log.Printf("Loading Grid Router Host from environment variable. New Value: %s", gridRouterHostENV)
	c.GridRouterHost = gridRouterHostENV

	if c.Clock == nil {
		c.Clock = clockwork.NewRealClock()
	}

	return nil
}

// Loads the data from a config file located at configs/config.yaml
func loadConfigFile() ([]byte, error) {
	return ioutil.ReadFile("configs/config.yaml")
}

// Loads the config.yaml file and reads it into a config struct
func (c *GridRebootConfig) LoadConfig() {
	yamlFile, err := loadConfigFile()
	if err != nil {
		log.Fatalf("Problem reading yaml file: %v", err)
	}

	c.rawData = yamlFile
}

// Takes the data from a config file and injects them into a GridRebootConfig struct
func (c *GridRebootConfig) populateStructFromConfigData() error {
	if c.rawData == nil || len(c.rawData) <= 0 {
		return errors.New("there was no configuration data loaded. Ensure you have configs/config.yaml")
	}
	return yaml.Unmarshal(c.rawData, &c)
}
