package config

import (
	"fmt"
	"github.com/jonboulle/clockwork"
	"github.com/stretchr/testify/assert"
	"os"
	"testing"
)

func TestNew(t *testing.T) {
	mockClock := clockwork.NewRealClock()

	var config GridRebootConfig
	config.Clock = mockClock

	// Create Test Data
	testRegion := "us-west-test"
	testTags := make(map[string]string)
	testTags["Service"] = "test"

	testHubTags := make(map[string]string)
	testHubTags["Name"] = "my-test-hub"

	// Create a Config File that will be used for mocking
	testConfig := getSampleConfigFile(testTags,
		testHubTags,
		testRegion,
		)

	// Load the data into the struct - we don't want to test actually opening the file
	config.rawData = testConfig

	expectedNodeConfig := Node{
		Tags:               testTags,
	}

	// Set up the expected EC2 Config Struct we should get back - it should have all our values
	expectedEC2Config := EC2{
		Nodes: expectedNodeConfig,
		Hub: Hub{
			Tags: testHubTags,
		},
		Region: testRegion,
	}

	// Set up the overall struct
	expectedGridConfig := GridRebootConfig{
		EC2: expectedEC2Config,
		rawData: testConfig,
		Clock:   mockClock,
	}

	// Now let's test it!
	config.Init()
	assert.Equal(t, expectedEC2Config, config.EC2)
	assert.Equal(t, expectedGridConfig, config)

	// "GRID_ENVIRONMENT" Environment Variable Should set EC2 Tag Environment
	testEnv := "envTest123"
	os.Setenv("GRID_ENVIRONMENT", testEnv)
	config.Init()
	assert.Equal(t, testEnv, config.EC2.Nodes.Tags["Environment"])

	t.Run("GRID_HUB_ENVIRONMENT", func (t *testing.T) {
		// Need to provide grid_router_host as well
		testEnvHost := "api.my.test.host"
		os.Setenv(envName_GridRouterHost, testEnvHost)
		defer os.Unsetenv(envName_GridRouterHost)

		t.Run("uses config when not provided", func (t *testing.T) {
			os.Unsetenv(envName_GridHubEnvironment)

			err := config.Init()
			assert.NoError(t, err)
			assert.Empty(t, config.EC2.Hub.Tags["Environment"])
		})

		t.Run("overrides config when provided", func (t *testing.T) {
			testENV := "test"
			os.Setenv(envName_GridHubEnvironment, testENV)
			defer os.Unsetenv(envName_GridHubEnvironment)

			err := config.Init()
			assert.NoError(t, err)
			assert.Equal(t, testENV, config.EC2.Hub.Tags["Environment"])
		})
	})

	t.Run("GRID_ROUTER_HOST", func (t *testing.T) {
		t.Run("fails when not provided", func (t *testing.T) {
			os.Unsetenv(envName_GridRouterHost)

			err := config.Init()
			assert.Error(t, err)
			assert.EqualError(t, err, "required env variable GRID_ROUTER_HOST not provided")
		})

		t.Run("passes when provided", func (t *testing.T) {
			testEnvHost := "api.my.test.host"
			os.Setenv(envName_GridRouterHost, testEnvHost)
			defer os.Unsetenv(envName_GridRouterHost)

			err := config.Init()
			assert.NoError(t, err)
		})
	})
}

func getSampleConfigFile(tags map[string]string, hubTags map[string]string,
	region string) []byte {

		// Creates a layout so that per tag, it is properly formatted into the file (indentation is important in yaml)
		tagLayout := ""
		for k, v := range tags {
			tagLayout += fmt.Sprintf("\n     %s: %s", k, v)
		}

		// Insert the hubs into the config
		hubLayout := ""
		for tagName, tagValue := range hubTags {
			hubLayout += fmt.Sprintf("\n        %s: \"%s\"", tagName, tagValue)
		}


		// Create the configuration file
		configLayout := `
ec2:
  nodes:
    tags: %s
  hub:
    tags: %s
  region: %s

`
		sampleConfig := fmt.Sprintf(configLayout, tagLayout, hubLayout, region)

		return []byte(sampleConfig)
}
