package ec2

import (
	"errors"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/ec2"
)

// Handler is used to interact with an AWS EC2 Service
type Handler struct {
	Config  *aws.Config
	Session *session.Session
	Service Service
}

// This interface defines the methods used from the EC2 SDK
// It was built as an interface to allow for better mocking
type Service interface {
	DescribeInstances(input *ec2.DescribeInstancesInput) (*ec2.DescribeInstancesOutput, error)
	RebootInstances(input *ec2.RebootInstancesInput) (*ec2.RebootInstancesOutput, error)
}

// Creates a new EC2 Service Connection, just from passing in the region to use
// This is essentially a shortcut to configure everything needed to communicate with the EC2 Service
func New(region string) (*Handler, error) {
	var h Handler

	// Set up Config
	config := aws.NewConfig()
	config.WithRegion(region)

	// Create the Session
	session, err := session.NewSession(config)
	if err != nil {
		return nil, err
	}

	// Create the Service
	svc := ec2.New(session)

	// Create the Struct Object and return it
	h = Handler{Config: config, Session: session, Service: svc}
	return &h, nil
}

// Gets all EC2 Instances from tags, and the max amount of pages to return
// Returns an EC2 Instance object
func (h *Handler) GetInstances(tags map[string]string, maxPages int64) ([]*ec2.Instance, error) {
	var instances []*ec2.Instance

	if len(tags) <= 0 {
		return instances, errors.New("must provide at least one tag to filter")
	}

	searchInput := GetSearchInputFromTags(tags, maxPages)

	result, err := h.Service.DescribeInstances(searchInput)
	instances = append(instances, getInstancesFromResult(result)...)

	if err != nil {
		printAWSError(err)
		return nil, err
	}

	return instances, nil
}

// Reboots a list of EC2 Instances
func (h *Handler) RebootInstances(instances []*ec2.Instance) (*ec2.RebootInstancesOutput, error) {
	// Give the RebootInstancesInput the list of the Instance IDs
	input := &ec2.RebootInstancesInput{
		InstanceIds: getInstanceIDsAsPointers(instances),
	}

	result, err := h.Service.RebootInstances(input)
	if err != nil {
		printAWSError(err)
		return nil, err
	}

	return result, nil
}
